/*
  PhasePictor - A program for displaying phaseplane of an ODE
  Copyright (c) 2007-2009 Kristóf Kály-Kullai
  This program is free software, it is distributed under the terms 
  of the GNU General Public License version 3 or (at your option) later.
  For details see the attached GPL-3.txt file.
  The software is provided "AS IS", WITHOUT WARRANTY of any kind.
*/
#if !defined(phasepictor_numeric)
 #define phasepictor_nuimeric
#include "mateklib.h"
#include <gtk/gtk.h>

#define MIN_DT 0.000001
/*maximum number of grid, leave as macro*/
#define MAX_GRID 64

typedef struct Parameter_t
               {gchar *name;
                double value;
               }parameter_t;


/***************************/
/*                         */
/*structure to store an ODE*/
/*                         */
/***************************/
typedef struct ODE
               {int dimension;
                int nparam;
                double global_dt;
                int changed;
                elem **equation;
                parameter_t *parameter;
               }ode;

/***************/
/*create an ode*/
/*0 on success,*/
/*-1 on error  */
/***************/
int ode_create(int dimension, int nparam);

/****************/
/*destroy an ode*/
/****************/
void ode_destroy();

/**********************/
/*sets a new equation */
/*return: 0 on success*/
/**********************/
int ode_set_equation(int i, gchar *equation_text);

/**********************/
/*gets i-th equation  */
/*return: 0 on success*/
/**********************/
int ode_get_equation(int i, gchar *equation_text);

/*************************************/
/*sets the name of the i-th parameter*/
/*return: 0 on success, !=0 on error */
/*************************************/
int ode_set_parameter_name(int i, const gchar *newname);

/**************************************/
/*sets the value of the i-th parameter*/
/**************************************/
void ode_set_parameter_value(int i, double newvalue);

/**************************************/
/*gets the value of the i-th parameter*/
/**************************************/
double ode_get_parameter_value(int i);

/************************************/
/*sets the name of the i-th variable*/
/*return: 0 on success, !=0 on error */
/************************************/
int ode_set_variable_name(int i, const gchar *newname);

/*****************************************/
/*sets the dt values for ALL trajectories*/
/*****************************************/
void ode_set_global_dt(double newdt);

/*****************************************/
/*gets the dt values for ALL trajectories*/
/*****************************************/
double ode_get_global_dt();

/*********************************/
/*                               */
/*structure to store a trajectory*/
/*                               */
/*********************************/

typedef struct Trajectory_t
               {gchar *name;
                double *initcond;
                double tmin, tmax, dt;
                double *dataplus, *dataminus;/*data for trajectory points*/
                size_t nplus,nminus;
                int changed;
                int color_id;
                int previplus, previminus;
                struct Trajectory_t *next, *prev;
               }trajectory_t;

#define NUM_TRAJ_COLORS 8
#define MAX_DOTSIZE 20

/******************************************/
/*adds a new trajectory to the phase plane*/
/*returns 0 on success                    */
/*-1 if unable to create new trajectory   */
/*1 if a trajectory with name exists      */
/******************************************/
int trajectory_add(const gchar *name, int color_id);

/******************************************/
/*removes a trajectory from the phaseplane*/
/******************************************/
int trajectory_delete(const gchar *name);

/****************************************************/
/*sets/gets the trajectories i-th initial coordinate*/
/****************************************************/
void trajectory_set_initcond(const gchar *name, int i, double value);

double trajectory_get_initcond(const gchar *name, int i);

/*****************************************/
/*sets/gets the time values of trajectory*/
/*****************************************/
void trajectory_set_tmin(const gchar *name, double tmin);

double trajectory_get_tmin(const gchar *name);

void trajectory_set_tmax(const gchar *name, double tmax);

double trajectory_get_tmax(const gchar *name);

void trajectory_set_dt(const gchar *name, double dt);

double trajectory_get_dt(const gchar *name);

/**********************************************************************/
/*gets changed value, !0 if uncalculated with the currently set values*/
/**********************************************************************/
int trajectory_get_changed(const gchar *name);

/**********************************/
/*gets the color of the trajectory*/
/**********************************/
void trajectory_get_color(const gchar *name, GdkColor *colTraj);

/****************************/
/*calculates the trajectory */
/*return with used tmax/tmin*/
/*if no trajectory: -1.0/1.0*/
/*on other error: <-1.0/>1.0*/
/****************************/
double trajectory_calculate_plus(const gchar *name);

double trajectory_calculate_minus(const gchar *name);

/*******************************/
/*stops the current calculation*/
/*******************************/
void trajectory_calculation_stop();

/**********************/
/*draws the trajectory*/
/**********************/
void trajectory_draw(const gchar *name, cairo_t *cr);

/***********************************************/
/*draws the dots at the end of trajectory      */
/*sizeplus, sizeminus:                         */
/*sizes of points at the ends of the trajectory*/
/***********************************************/
void trajectory_dot_draw(const gchar *name, cairo_t *cr,
                         double sizeplus, double sizeminus);


/*******************************/
/*                             */
/*functions for the phase plane*/
/*                             */
/*******************************/

/****************************/
/*initializes the phaseplane*/
/* returns 0 on success     */
/****************************/
int phaseplane_init(int dimension, int nparam);

/*************************/
/*finishes the phaseplane*/
/*************************/
void phaseplane_destroy();

/*********************************************/
/*sets the colors used to draw the phaseplane*/
/*********************************************/
void phaseplane_set_trajectory_color(double r, double g, double b);

void phaseplane_set_draw_color_traj(int dct);

void phaseplane_set_background_color(double r, double g, double b);

void phaseplane_set_axis_color(double r, double g, double b);

void phaseplane_set_dirfield_color(double r, double g, double b);

/***********************************/
/*sets the window of the phaseplane*/
/***********************************/
void phaseplane_set_window(double xmin, double xmax, double ymin, double ymax);

/***********************************/
/*gets the window of the phaseplane*/
/***********************************/
void phaseplane_get_window(double *xmin, double *xmax, double *ymin, double *ymax);

/*********************************************/
/*sets the grid number for the directionfield*/
/*********************************************/
void phaseplane_set_dirfield_grid(int grid);

/*********************************************/
/*gets the grid number for the directionfield*/
/*********************************************/
int phaseplane_get_dirfield_grid();

/***********************************/
/*sets the time frame to draw until*/
/***********************************/
void phaseplane_set_time_frame(double tmin, double tmax);

/***********************************/
/*gets the time frame to draw until*/
/***********************************/
void phaseplane_get_time_frame(double *tmin, double *tmax);

/************************************************/
/*returns: 1, if any trajectory has changed set,*/
/*         0 otherwise                          */
/************************************************/
int phaseplane_trajectory_changed();

/************************/
/*draws a directionfield*/
/************************/
void phaseplane_draw_dirfield(cairo_t *cr);

/************************************/
/*redraws the whole phaseplane to cr*/
/************************************/
void phaseplane_draw(cairo_t *cr, double width, double height);

/**********************************************/
/*draws trajectories from startplus/startminus*/
/**********************************************/
void phaseplane_draw_delta(cairo_t *cr, double width, double height,
                           double startplus, double startminus);

/***************************************/
/*draws dots to the end of trajectories*/
/***************************************/
void phaseplane_draw_dots(cairo_t *cr, double width, double height,
                          double sizeplus, double sizeminus);

/**************************************/
/*sets the variables shown on the axis*/
/*-1 indicates time, 0 is the 1st var.*/
/**************************************/
void phaseplane_set_shown_variables(int newix, int newiy);

/***********************************/
/*Saves the current ODE, parameters*/
/*and trajectories to a text file. */
/*return: 0 on success, -1 on error*/
/***********************************/
int phaseplane_save_to_file(FILE *outfile);

#endif
