/*
  PhasePictor - A program for displaying phaseplane of an ODE
  Copyright (c) 2007-2009 Kristóf Kály-Kullai
  This program is free software, it is distributed under the terms 
  of the GNU General Public License version 3 or (at your option) later.
  For details see the attached GPL-3.txt file.
  The software is provided "AS IS", WITHOUT WARRANTY of any kind.
*/
#include "phasepictor_numeric.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

/*global phase plane variables*/
ode o;
rk_t *rk;
trajectory_t *trajectory=0;
int grid_num;
double *grid_data;
int numtraj;/*number of trajectories*/
double pp_xmin, pp_xmax, pp_ymin, pp_ymax, cr_ymax;
double limit_mult, lim_xmin, lim_xmax, lim_ymin, lim_ymax;/*limits of cutting*/
double draw_tmin, draw_tmax;/*time frame to draw*/
double rtraj, gtraj, btraj;
int draw_color_traj;
double rtraj_color[NUM_TRAJ_COLORS]={1.0, 1.0, 1.0, 1.0, 0.0, 0.0, 0.0, 1.0};
double gtraj_color[NUM_TRAJ_COLORS]={1.0, 0.0, 0.5, 1.0, 1.0, 1.0, 0.0, 0.0};
double btraj_color[NUM_TRAJ_COLORS]={1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0};
double rbackgr, gbackgr, bbackgr;
double raxis, gaxis, baxis;
double rdirf, gdirf, bdirf;
double xscale, xoffset, yscale, yoffset;
int ix,iy;/*indexes of variables drawing as x, y; -1 indicates time*/
int calculation_stop=0;/*if !=0, stops the current trajectory calculation*/

/*default names to valtozok[i].nev*/
gchar default_names[MAX_VALT-2][8];


/*MACROS*/
/*convert coordinates from phaseplane to cairo system*/
#define PP2CR_X(x) ( xscale*(x)+xoffset )
#define PP2CR_Y(y) ( cr_ymax-yscale*(y)-yoffset )

/*********************/
/*                   */
/* F U N C T I O N S */
/*                   */
/*********************/

/*function for RK method*/
int function(double *y,double t,double *fyt)
{int i;

 for(i=0;i<o.dimension;i++)
    {valtozok[2+i].ertek=y[i];
    }
 valtozok[1].ertek=t;
 for(i=0;i<o.dimension;i++)
    {fyt[i]=ertekel(o.equation[i]);
     if(materrno){return materrno;}
    }
 return 0;
}

/***************/
/*create an ode*/
/*0 on success,*/
/*-1 on error  */
/***************/
int ode_create(int dimension, int nparam)
{int i;

 if(dimension+nparam>MAX_VALT-2){return -1;}
 if(dimension<1){return -1;}
 if(nparam<0){return -1;}
 o.equation=(elem**)calloc(dimension, sizeof(elem*));
 if(o.equation==NULL)
   {return -1;}
 o.parameter=(parameter_t*)calloc(nparam, sizeof(parameter_t));
 if(o.parameter==NULL)
   {free(o.equation);
    return -1;
   }
 o.dimension=dimension;
 o.nparam=nparam;
 o.global_dt=0.0;/*means no global_dt set*/
 
 /*initialize an rk_t*/
 rk=rk_create(o.dimension);
 if(rk==NULL)
   {free(o.parameter);
    free(o.equation);
    return -1;
   }
 rk_set_function(rk, function);
 
 /*initialize mateklib symbolic calculation*/
 initmat();
 valtozok[1].nev="t";
 valtozok[1].ertek=0.0;
 /*valtozok[0]=pi, valtozok[1]=t, then variables, then parameters*/
 for(i=2;i<2+o.dimension+o.nparam;i++)
    {valtozok[i].ertek=0.0;}
 for(i=0;i<o.dimension;i++)
    {sprintf(default_names[i],"x%d",i);
     valtozok[2+i].nev=g_strdup(default_names[i]);
    }
 for(i=0;i<o.nparam;i++)
    {sprintf(default_names[i],"p%d",i);
     valtozok[2+o.dimension+i].nev=default_names[o.dimension+i];
     o.parameter[i].name=g_strdup(default_names[o.dimension+i]);
    }
 return 0;
}

/****************/
/*destroy an ode*/
/****************/
void ode_destroy()
{int i;

 rk_destroy(rk);
 
 free(o.parameter);
 for(i=0;i<o.dimension;i++)
    {torol(o.equation[i]);}
 free(o.equation);
 return;
}

/**********************/
/*sets a new equation */
/*return: 0 on success*/
/**********************/
int ode_set_equation(int i, gchar *equation_text)
{int e;
 trajectory_t *ft;

 if( (i<0)||(i>=o.dimension) ){return -1;}
 if(o.equation[i]){torol(o.equation[i]);}
 ft=trajectory;
 while(ft)
    {ft->changed=1;
     ft=ft->next;
    }
 e=olvas((char *)(equation_text), &o.equation[i]);
 if( ( (int)strlen((const char*)equation_text)>e ) || (o.equation[i]==NULL) )
   {if( (o.equation[i]==NULL)||(e==0) ){return -1;}
    else{return e;}
   }
 else{return 0;}
}

/**********************/
/*gets i-th equation  */
/*return: 0 on success*/
/**********************/
int ode_get_equation(int i, gchar *equation_text)
{int ret;
 
 if( (i<0)||(i>=o.dimension) ){return -1;}
 ret=kiir(o.equation[i], (char*)equation_text);
 return (ret<0) ? ret : 0;
}

/*************************************/
/*sets the name of the i-th parameter*/
/*return: 0 on success, !=0 on error */
/*************************************/
int ode_set_parameter_name(int i, const gchar *newname)
{int j;

 if( (i<0)||(i>=o.nparam) ){return -1;}
 if(newname[0]==0){return -1;}
 for(j=0;j<i;j++)
    {if( strcmp(valtozok[2+o.dimension+j].nev, (char*)newname) == 0 )
       {return j+1;}
    }
 for(j=i+1;j<o.nparam;j++)
    {if( strcmp(valtozok[2+o.dimension+j].nev, (char*)newname) == 0 )
       {return j+1;}
    }
 if(o.parameter[i].name){g_free(o.parameter[i].name);}
 o.parameter[i].name=g_strdup(newname);
 if(o.parameter[i].name==NULL){return -1;}
 valtozok[2+o.dimension+i].nev=o.parameter[i].name;
 return 0;
}

/**************************************/
/*sets the value of the i-th parameter*/
/**************************************/
void ode_set_parameter_value(int i, double newvalue)
{trajectory_t *ft;
 
 if( (i<0)||(i>=o.nparam) )return;
 o.parameter[i].value=newvalue;
 valtozok[2+o.dimension+i].ertek=newvalue;
 ft=trajectory;
 while(ft)
    {ft->changed=1;
     ft=ft->next;
    }
 return;
}

/**************************************/
/*gets the value of the i-th parameter*/
/**************************************/
double ode_get_parameter_value(int i)
{
 if( (i<0)||(i>=o.nparam) )return 0.0;
 return o.parameter[i].value;
}

/************************************/
/*sets the name of the i-th variable*/
/*return: 0 on success, !=0 on error */
/************************************/
int ode_set_variable_name(int i, const gchar *newname)
{int j;

 if( (i<0)||(i>=o.dimension) ){return -1;}
 if(newname[0]==0){return -1;}
 for(j=0;j<i-1;j++)
    {if( strcmp(valtozok[2+j].nev, (char*)newname) == 0 )
       {return j+1;}
    }
 for(j=i+1;j<o.nparam;j++)
    {if( strcmp(valtozok[2+j].nev, (char*)newname) == 0 )
       {return j+1;}
    }
 if(valtozok[2+i].nev){g_free(valtozok[2+i].nev);}
 valtozok[2+i].nev=g_strdup(newname);
 if(valtozok[2+i].nev==NULL){return -1;}
 return 0;
}

/*****************************************/
/*sets the dt values for ALL trajectories*/
/*****************************************/
void ode_set_global_dt(double newdt)
{trajectory_t *ft;

 if(newdt<0.0){return;}
 if((newdt<MIN_DT)&&(newdt!=0.0)){o.global_dt=MIN_DT;}
  else{o.global_dt=newdt;}
 /*0.0 means no global dt*/
 if(o.global_dt>0.0)
   {ft=trajectory;
    while(ft)
       {if(ft->dt!=newdt)
          {ft->dt=newdt;
           ft->changed=1;
          }
        ft=ft->next;
       }
   }
}

/*****************************************/
/*gets the dt values for ALL trajectories*/
/*****************************************/
double ode_get_global_dt()
{return o.global_dt;
}

/******************************************/
/*returns the trajectory with name or NULL*/
/******************************************/
trajectory_t *trajectory_find(const gchar *name)
{trajectory_t *ft;

 if(name==NULL){return NULL;}
 if(name[0]=='\0'){return NULL;}
 ft=trajectory;
 while(ft)
    {if( strcmp(ft->name, name)==0 )
       {break;}
     ft=ft->next;
    }
 return ft;
}

/******************************************/
/*adds a new trajectory to the phase plane*/
/*returns 0 on success                    */
/*-1 if unable to create new trajectory   */
/*1 if a trajectory with name exists      */
/******************************************/
int trajectory_add(const gchar *name, int color_id)
{trajectory_t *ft;
 int MDS2;

 if(name[0]==0){return -1;}
 ft=trajectory_find(name);
 if(ft!=NULL){return 1;}
 ft=trajectory;
 MDS2=(MAX_DOTSIZE+2)*(MAX_DOTSIZE+2);
 /*if first element in the list*/
 if(ft==NULL)
   {ft=(trajectory_t *)calloc(1, sizeof(trajectory_t));
     if(ft==NULL){return -1;}
    ft->initcond=calloc(o.dimension, sizeof(double));
     if(ft->initcond==NULL){free(ft); return -1;}
    ft->name=g_strdup(name);
    ft->next=NULL;
    ft->prev=NULL;
    ft->nplus=0;
    ft->nminus=0;
    if(o.global_dt>0.0){ft->dt=o.global_dt;}
    ft->changed=0;
    if(color_id<0){ft->color_id=(-color_id)%NUM_TRAJ_COLORS;}
     else{ft->color_id=color_id%NUM_TRAJ_COLORS;}
    ft->previplus=0;
    ft->previminus=0;
    trajectory=ft;
    numtraj++;
    return 0;
   }
 
 /*append a new element to the list*/
 while(ft->next)
    {ft=ft->next;
    }
 ft->next=(trajectory_t *)calloc(1, sizeof(trajectory_t));
  if(ft->next==NULL){return -1;}
 ft->next->initcond=calloc(o.dimension, sizeof(double));
  if(ft->next->initcond==NULL){free(ft->next); return -1;}
 ft->next->prev=ft;
 ft=ft->next;
 ft->name=g_strdup(name);
 ft->next=NULL;
 ft->nplus=0;
 ft->nminus=0;
 if(o.global_dt>0.0){ft->dt=o.global_dt;}
 ft->changed=0;
 ft->color_id=color_id%NUM_TRAJ_COLORS;
 ft->previplus=0;
 ft->previminus=0;
 numtraj++;
 return 0;
}

/******************************************/
/*removes a trajectory from the phaseplane*/
/******************************************/
int trajectory_delete(const gchar *name)
{trajectory_t *ft;

 ft=trajectory_find(name);
 if(ft==NULL){return -1;}
 if(ft->prev){ft->prev->next=ft->next;}
 if(ft->next){ft->next->prev=ft->prev;}
 if(ft==trajectory){trajectory=ft->next;}
 free(ft->dataplus);
 free(ft->dataminus);
 g_free(ft->name);
 free(ft->initcond); 
 free(ft);
 numtraj--;
 return 0;
}

/****************************************************/
/*sets/gets the trajectories i-th initial coordinate*/
/****************************************************/
void trajectory_set_initcond(const gchar *name, int i, double value)
{trajectory_t *ft;

 ft=trajectory_find(name);
 if(ft==NULL){return;}
 if( (i<0)||(i>=o.dimension) ){return;}
 if(ft->initcond[i]!=value)
   {ft->initcond[i]=value;
    ft->changed=1;
   }
 return;
}

double trajectory_get_initcond(const gchar *name, int i)
{trajectory_t *ft;

 ft=trajectory_find(name);
 if(ft==NULL){return 0.0;}
 if( (i<0)||(i>=o.dimension) ){return 0.0;}
 return ft->initcond[i];
}

/*****************************************/
/*sets/gets the time values of trajectory*/
/*****************************************/
void trajectory_set_tmin(const gchar *name, double tmin)
{trajectory_t *ft;
 double old_value;

 ft=trajectory_find(name);
 if(ft==NULL){return;}
 old_value=ft->tmin;
 ft->tmin=MINIMUM(0, tmin);
 if(old_value!=ft->tmin){ft->changed=1;}
}

double trajectory_get_tmin(const gchar *name)
{trajectory_t *ft;

 ft=trajectory_find(name);
 if(ft==NULL){return 0.0;}
 return ft->tmin;
}

void trajectory_set_tmax(const gchar *name, double tmax)
{trajectory_t *ft;
 double old_value;

 ft=trajectory_find(name);
 if(ft==NULL){return;}
 old_value=ft->tmax;
 ft->tmax=MAXIMUM(0, tmax);
 if(old_value!=ft->tmax){ft->changed=1;}
}

double trajectory_get_tmax(const gchar *name)
{trajectory_t *ft;

 ft=trajectory_find(name);
 if(ft==NULL){return 0.0;}
 return ft->tmax;
}

void trajectory_set_dt(const gchar *name, double dt)
{trajectory_t *ft;
 double old_value;

 if(dt<0.0){return;}
 ft=trajectory_find(name);
 if(ft==NULL){return;} 
 old_value=ft->dt;
 ft->dt=(o.global_dt>=MIN_DT) ? o.global_dt : ((dt>MIN_DT) ? dt : MIN_DT);
 if(old_value!=ft->dt){ft->changed=1;}
}

double trajectory_get_dt(const gchar *name)
{trajectory_t *ft;

 ft=trajectory_find(name);
 if(ft==NULL){return 0.0;}
 return ft->dt;
}

/**********************************************************************/
/*gets changed value, !0 if uncalculated with the currently set values*/
/**********************************************************************/
int trajectory_get_changed(const gchar *name)
{trajectory_t *ft;

 ft=trajectory_find(name);
 if(ft==NULL){return 0;}
 return ft->changed;
}

/**********************************/
/*gets the color of the trajectory*/
/**********************************/
void trajectory_get_color(const gchar *name, GdkColor *colTraj)
{trajectory_t *ft;
 
 ft=trajectory_find(name);
 if(ft==NULL){return;}
 if(ft->color_id!=0)
   {colTraj->red=(guint16)(rtraj_color[ft->color_id]*65535);
    colTraj->green=(guint16)(gtraj_color[ft->color_id]*65535);
    colTraj->blue=(guint16)(btraj_color[ft->color_id]*65535);
   }
 else
   {colTraj->red=(guint16)(rtraj*65535);
    colTraj->green=(guint16)(gtraj*65535);
    colTraj->blue=(guint16)(btraj*65535);
   }
 return;
}

/****************************/
/*calculates the trajectory */
/*return with used tmax/tmin*/
/*if no trajectory: -1.0/1.0*/
/*on other error: <-1.0/>1.0*/
/****************************/
double trajectory_calculate_plus(const gchar *name)
{trajectory_t *ft;
 double d;
 size_t s;

 ft=trajectory_find(name);
 if(ft==NULL){return -1.0;}
 calculation_stop=0;
 /*check variables*/
 s=UINT_MAX/o.dimension-1;
 d=MINIMUM(fabs(ft->tmax/ft->dt), (double)(s));
 ft->nplus=(size_t)(d+0.5);
 ft->tmax=ft->dt*((double)(ft->nplus));
 ft->nplus++;
 
 /*allocate memory*/
 if(ft->nplus)
   {ft->dataplus=(double *)realloc(ft->dataplus, o.dimension*ft->nplus*sizeof(double));
    if(ft->dataplus==NULL)
      {ft->nplus=0;
       return -2.0;
      }
   }
 else{return 0.0;}
 
 /*calculate trajectories with RK4*/
 for(s=0;s<o.dimension;s++)
    {ft->dataplus[s]=ft->initcond[s];}
 d=0;
 for(s=1;s<ft->nplus;s++)
    {while(gtk_events_pending())
        {gtk_main_iteration();}
     if(calculation_stop)
       {calculation_stop=0;
        ft->tmax=d;
        ft->nplus=s;
        return d;
       }
     if( rk_iteration(rk, ft->dataplus+(s-1)*o.dimension, 
                               d, ft->dt, ft->dataplus+s*o.dimension) )
       {ft->tmax=d;
        ft->nplus=s;
        break;
       }
     d+=ft->dt;
    }
 ft->changed|=0x2;
 if( ft->changed & 0x6 )
   {ft->changed=0;}
 return ft->tmax;
}

double trajectory_calculate_minus(const gchar *name)
{trajectory_t *ft;
 double d;
 size_t s;

 ft=trajectory_find(name);
 if(ft==NULL){return 1.0;}
 calculation_stop=0;
 /*check variables*/
 s=UINT_MAX/o.dimension-1;
 d=MINIMUM(fabs(ft->tmin/ft->dt), (double)(s));
 ft->nminus=(size_t)(d+0.5);
 ft->tmin=-ft->dt*((double)(ft->nminus));
 ft->nminus++;
 
 /*allocate memory*/
 if(ft->nminus)
   {ft->dataminus=(double *)realloc(ft->dataminus, o.dimension*ft->nminus*sizeof(double));
    if(ft->dataminus==NULL)
      {ft->nminus=0;
       return 2.0;
      }
   }
 else{return 0.0;}
 
 /*calculate trajectories with RK4*/
 for(s=0;s<o.dimension;s++)
    {ft->dataminus[s]=ft->initcond[s];}
 d=0;
 for(s=1;s<ft->nminus;s++)
    {while(gtk_events_pending())
        {gtk_main_iteration();}
     if(calculation_stop)
       {calculation_stop=0;
        ft->tmin=d;
        ft->nminus=s;
        return d;
       }
     if( rk_iteration(rk, ft->dataminus+(s-1)*o.dimension, 
                               d, -ft->dt, ft->dataminus+s*o.dimension) )
       {ft->tmin=d;
        ft->nminus=s;
        break;
       }
     d-=ft->dt;
    }
 ft->changed|=0x4;
 if( ft->changed & 0x6 )
   {ft->changed=0;}
 return ft->tmin;
}

/*******************************/
/*stops the current calculation*/
/*******************************/
void trajectory_calculation_stop()
{
 calculation_stop=1;
}

/**********************/
/*draws the trajectory*/
/**********************/
void trajectory_draw_poi(trajectory_t *ft, cairo_t *cr,
                         double startplus, double startminus)
{int i, itmin,itmax;

 if(ft==NULL){return;}
 itmax=(int)(draw_tmax/ft->dt);
  if(itmax<INT_MAX){itmax++;}
 itmin=-(int)(draw_tmin/ft->dt);
  if(itmin<INT_MAX){itmin++;}
 /*set up cairo*/
 /*cairo_set_antialias(cr, CAIRO_ANTIALIAS_NONE);*/
 if((draw_color_traj!=0)&&(ft->color_id!=0))
   {cairo_set_source_rgb (cr, rtraj_color[ft->color_id],
                          gtraj_color[ft->color_id], btraj_color[ft->color_id]);
   }
 else
    {cairo_set_source_rgb (cr, rtraj, gtraj, btraj);}
 cairo_set_line_width(cr,1.0);
 cairo_set_line_join(cr,CAIRO_LINE_JOIN_MITER);
 cairo_set_line_cap(cr,CAIRO_LINE_CAP_BUTT);
 
 /*drawing plus part*/
 if( (ft->nplus)&&(startplus<=ft->tmax) )
   {if(startplus>0.0)
      {i=(int)(startplus/ft->dt);}
    else {i=0;}
    if( (ft->dataplus[i*o.dimension+ix]<=lim_xmax)&&(ft->dataplus[i*o.dimension+ix]>=lim_xmin)
      &&(ft->dataplus[i*o.dimension+iy]<=lim_ymax)&&(ft->dataplus[i*o.dimension+iy]>=lim_ymin) )
      {/*1st point*/
       cairo_move_to( cr, PP2CR_X(ft->dataplus[i*o.dimension+ix]), PP2CR_Y(ft->dataplus[i*o.dimension+iy]) );
       cairo_rel_line_to(cr, 1, 0);
       cairo_move_to( cr, PP2CR_X(ft->dataplus[i*o.dimension+ix]), PP2CR_Y(ft->dataplus[i*o.dimension+iy]) );
       i++;
      }
    else
      {/*go to 1st inside*/
       i++;
       while(i<MINIMUM(ft->nplus,itmax))
         {if( (ft->dataplus[i*o.dimension+ix]<=lim_xmax)&&(ft->dataplus[i*o.dimension+ix]>=lim_xmin)
            &&(ft->dataplus[i*o.dimension+iy]<=lim_ymax)&&(ft->dataplus[i*o.dimension+iy]>=lim_ymin) )
            {break;}
          else {i++;}
         }
       i--;
       cairo_move_to(cr, PP2CR_X(ft->dataplus[i*o.dimension+ix]), PP2CR_Y(ft->dataplus[i*o.dimension+iy]) );
       i++;
      }
    /*draw the trajectory*/
    for(; i<MINIMUM(ft->nplus,itmax); i++)
       {cairo_line_to(cr, PP2CR_X(ft->dataplus[i*o.dimension+ix]), PP2CR_Y(ft->dataplus[i*o.dimension+iy]) );
        /*go to the last point before inside view*/
        if( (ft->dataplus[i*o.dimension+ix]>lim_xmax)||(ft->dataplus[i*o.dimension+ix]<lim_xmin)
          ||(ft->dataplus[i*o.dimension+iy]>lim_ymax)||(ft->dataplus[i*o.dimension+iy]<lim_ymin) )
          {while(i<MINIMUM(ft->nplus,itmax))
              {i++;
               if( (ft->dataplus[i*o.dimension+ix]<=lim_xmax)&&(ft->dataplus[i*o.dimension+ix]>=lim_xmin)
                 &&(ft->dataplus[i*o.dimension+iy]<=lim_ymax)&&(ft->dataplus[i*o.dimension+iy]>=lim_ymin) )
                 {i--; break;}
              }
          }
        cairo_move_to(cr, PP2CR_X(ft->dataplus[i*o.dimension+ix]), PP2CR_Y(ft->dataplus[i*o.dimension+iy]) );
       }
    cairo_stroke(cr);
    ft->previplus=i-1;
   }
 
 /*drawing minus part*/
 if( (ft->nminus)&&(startminus>=ft->tmin) )
   {if(startminus<0.0)
      {i=(int)(-startminus/ft->dt);}
    else {i=0;}
    if( (ft->dataminus[i*o.dimension+ix]<=lim_xmax)&&(ft->dataminus[i*o.dimension+ix]>=lim_xmin)
      &&(ft->dataminus[i*o.dimension+iy]<=lim_ymax)&&(ft->dataminus[i*o.dimension+iy]>=lim_ymin) )
      {/*1st point*/
       cairo_move_to( cr, PP2CR_X(ft->dataminus[i*o.dimension+ix]), PP2CR_Y(ft->dataminus[i*o.dimension+iy]) );
       cairo_rel_line_to(cr, 1, 0);
       cairo_move_to( cr, PP2CR_X(ft->dataminus[i*o.dimension+ix]), PP2CR_Y(ft->dataminus[i*o.dimension+iy]) );
       i++;
      }
    else
      {/*go to 1st inside*/
       i++;
       while(i<MINIMUM(ft->nminus,itmin))
         {if( (ft->dataminus[i*o.dimension+ix]<=lim_xmax)&&(ft->dataminus[i*o.dimension+ix]>=lim_xmin)
            &&(ft->dataminus[i*o.dimension+iy]<=lim_ymax)&&(ft->dataminus[i*o.dimension+iy]>=lim_ymin) )
            {break;}
          else {i++;}
         }
       i--;
       cairo_move_to(cr, PP2CR_X(ft->dataminus[i*o.dimension+ix]), PP2CR_Y(ft->dataminus[i*o.dimension+iy]) );
       i++;
      }
    /*draw the trajectory*/
    for(; i<MINIMUM(ft->nminus,itmin); i++)
       {cairo_line_to(cr, PP2CR_X(ft->dataminus[i*o.dimension+ix]), PP2CR_Y(ft->dataminus[i*o.dimension+iy]) );
        /*go to the last point before inside view*/
        if( (ft->dataminus[i*o.dimension+ix]>lim_xmax)||(ft->dataminus[i*o.dimension+ix]<lim_xmin)
          ||(ft->dataminus[i*o.dimension+iy]>lim_ymax)||(ft->dataminus[i*o.dimension+iy]<lim_ymin) )
          {while(i<MINIMUM(ft->nminus,itmin))
              {i++;
               if( (ft->dataminus[i*o.dimension+ix]<=lim_xmax)&&(ft->dataminus[i*o.dimension+ix]>=lim_xmin)
                 &&(ft->dataminus[i*o.dimension+iy]<=lim_ymax)&&(ft->dataminus[i*o.dimension+iy]>=lim_ymin) )
                 {i--; break;}
              }
          }
        cairo_move_to(cr, PP2CR_X(ft->dataminus[i*o.dimension+ix]), PP2CR_Y(ft->dataminus[i*o.dimension+iy]) );
       }
    cairo_stroke(cr);
    ft->previminus=i-1;
   }
 return;
}

void trajectory_draw(const gchar *name, cairo_t *cr)
{trajectory_t *ft;

 /*search the trajectory*/
 ft=trajectory_find(name);
 if(ft==NULL){return;}
 trajectory_draw_poi(ft, cr, 0.0, 0.0);
 return;
}

/***********************************************/
/*draws the dots at the end of trajectory      */
/*sizeplus, sizeminus:                         */
/*sizes of points at the ends of the trajectory*/
/***********************************************/
void trajectory_dot_draw_poi(trajectory_t *ft, cairo_t *cr,
                             double sizeplus, double sizeminus)
{double splus, smin, x,y;
 int delta;
 
 /*set up cairo*/
 if((draw_color_traj!=0)&&(ft->color_id!=0))
   {cairo_set_source_rgb (cr, rtraj_color[ft->color_id],
                          gtraj_color[ft->color_id], btraj_color[ft->color_id]);
   }
 else
    {cairo_set_source_rgb (cr, rtraj, gtraj, btraj);}
 cairo_set_line_width(cr,1.0);
 cairo_set_line_join(cr,CAIRO_LINE_JOIN_MITER);
 cairo_set_line_cap(cr,CAIRO_LINE_CAP_BUTT);
 
 delta=MAX_DOTSIZE/2+1;
 if(ft->nplus)
   {x=ft->dataplus[ft->previplus*o.dimension+ix];
    y=ft->dataplus[ft->previplus*o.dimension+iy];
    if( (sizeplus>0.0)&&(x>=pp_xmin)&&(x<=pp_xmax)&&(y>=pp_ymin)&&(y<=pp_ymax) )
      {x=PP2CR_X(x);
       y=PP2CR_Y(y);
       splus=MINIMUM(sizeplus,MAX_DOTSIZE);
       cairo_arc(cr, x, y, 0.5*splus, 0, 6.284);
       cairo_fill(cr);
      }
   }
 if(ft->nminus)
   {x=ft->dataminus[ft->previminus*o.dimension+ix];
    y=ft->dataminus[ft->previminus*o.dimension+iy];
    if( (sizeminus>0.0)&&(x>=pp_xmin)&&(x<=pp_xmax)&&(y>=pp_ymin)&&(y<=pp_ymax) )
      {x=PP2CR_X(x);
       y=PP2CR_Y(y);
       smin=MINIMUM(sizeminus,MAX_DOTSIZE);
       cairo_rectangle(cr, x-0.5*smin, y-0.5*smin, smin, smin);
       cairo_fill(cr);
      }
   }
 return;
}

void trajectory_dot_draw(const gchar *name, cairo_t *cr,
                         double sizeplus, double sizeminus)
{trajectory_t *ft;

 /*search the trajectory*/
 ft=trajectory_find(name);
 if(ft==NULL){return;}
 trajectory_dot_draw_poi(ft, cr, sizeplus, sizeminus);
 return;
}


/*******************************/
/*                             */
/*functions for the phase plane*/
/*                             */
/*******************************/

/****************************/
/*initializes the phaseplane*/
/* returns 0 on success     */
/****************************/
int phaseplane_init(int dimension, int nparam)
{
 if(dimension<1){return -1;}
 grid_num=0;
 ix=0; iy=1;
 pp_xmin=-3.0; pp_xmax=3.0; pp_ymin=-3.0; pp_ymax=3.0;
 draw_tmin=-MIN_DT*INT_MAX; draw_tmax=MIN_DT*INT_MAX;
 limit_mult=0.05;
 lim_xmin=-3.15; lim_xmax=3.15; lim_ymin=-3.15; lim_ymax=3.15;
 rtraj=1; gtraj=1; btraj=1;
 draw_color_traj=1;
 rbackgr=0; gbackgr=0; bbackgr=0;
 raxis=0.8; gaxis=0.8; baxis=0.8;
 rdirf=0; gdirf=1; bdirf=1;
 numtraj=0;
 if( ode_create(dimension, MAXIMUM(0,nparam)) == -1 ){return -1;}
 grid_data=(double*)calloc(MAX_GRID*MAX_GRID*2,sizeof(double));
 if(grid_data==NULL)
   {return -2;}
 
 return 0;
}

/*************************/
/*finishes the phaseplane*/
/*************************/
void phaseplane_destroy()
{
 while(trajectory)
    {trajectory_delete(trajectory->name);
    }
 
 ode_destroy();
 
 return;
}

/*********************************************/
/*sets the colors used to draw the phaseplane*/
/*********************************************/
void phaseplane_set_trajectory_color(double r, double g, double b)
{
 rtraj=MINIMUM(MAXIMUM(0.0,r),1.0);
 gtraj=MINIMUM(MAXIMUM(0.0,g),1.0);
 btraj=MINIMUM(MAXIMUM(0.0,b),1.0);
 return;
}

void phaseplane_set_draw_color_traj(int dct)
{
 draw_color_traj=dct;
 return;
}

void phaseplane_set_background_color(double r, double g, double b)
{
 rbackgr=MINIMUM(MAXIMUM(0.0,r),1.0);
 gbackgr=MINIMUM(MAXIMUM(0.0,g),1.0);
 bbackgr=MINIMUM(MAXIMUM(0.0,b),1.0);
 return;
}

void phaseplane_set_axis_color(double r, double g, double b)
{
 raxis=MINIMUM(MAXIMUM(0.0,r),1.0);
 gaxis=MINIMUM(MAXIMUM(0.0,g),1.0);
 baxis=MINIMUM(MAXIMUM(0.0,b),1.0);
 return;
}

void phaseplane_set_dirfield_color(double r, double g, double b)
{
 rdirf=MINIMUM(MAXIMUM(0.0,r),1.0);
 gdirf=MINIMUM(MAXIMUM(0.0,g),1.0);
 bdirf=MINIMUM(MAXIMUM(0.0,b),1.0);
 return;
}

/***********************************/
/*sets the window of the phaseplane*/
/***********************************/
void phaseplane_set_window(double xmin, double xmax, double ymin, double ymax)
{
 if(1e-7>=xmax-xmin){return;}
 if(1e-7>=ymax-ymin){return;}
 pp_xmin=xmin;
 pp_xmax=xmax;
 pp_ymin=ymin;
 pp_ymax=ymax;
 lim_xmin=pp_xmin-(pp_xmax-pp_xmin)*limit_mult;
 lim_xmax=pp_xmax+(pp_xmax-pp_xmin)*limit_mult;
 lim_ymin=pp_ymin-(pp_ymax-pp_ymin)*limit_mult;
 lim_ymax=pp_ymax+(pp_ymax-pp_ymin)*limit_mult;
 return;
}

/***********************************/
/*gets the window of the phaseplane*/
/***********************************/
void phaseplane_get_window(double *xmin, double *xmax, double *ymin, double *ymax)
{
 *xmin=pp_xmin;
 *xmax=pp_xmax;
 *ymin=pp_ymin;
 *ymax=pp_ymax;
 return;
}

/*********************************************/
/*gets the grid number for the directionfield*/
/*********************************************/
int phaseplane_get_dirfield_grid()
{
 return grid_num;
}

/*********************************************/
/*sets the grid number for the directionfield*/
/*********************************************/
void phaseplane_set_dirfield_grid(int grid)
{double vx, vy, dgridx, dgridy, dgrid, a;
 int i,j;
 
 if(grid_data)
   {grid_num=MINIMUM(MAXIMUM(0, grid), MAX_GRID);
   }
 else
   {grid_num=0;
   }
 if(grid_num<=0){return;}
 /*calculate the distance of grid points*/
 dgridx=(pp_xmax-pp_xmin)/((double)grid_num);
 dgridy=(pp_ymax-pp_ymin)/((double)grid_num);
 dgrid=MINIMUM(dgridx,dgridy)*0.5;
 /*drawing the grid points*/
 for(i=0;i<grid_num;i++)
    for(j=0;j<grid_num;j++)
       {/*calculating the direction field*/
        /*@ other variables and time set to the section value*/
        valtozok[1].ertek=0.0;
        valtozok[2+ix].ertek=pp_xmin+dgridx*((double)i+0.5);
        valtozok[2+iy].ertek=pp_ymin+dgridy*((double)j+0.5);
        vx=ertekel(o.equation[ix]);
        if(materrno)
          {vx=0.0; vy=0.0;}
        else
          {vy=ertekel(o.equation[iy]);
           if(materrno){vx=0.0; vy=0.0;}
          }
        a=sqrt(vx*vx+vy*vy);
        if(fabs(a)>1e-7){a=atan(a)/(M_PI_2*a);}
         else{a=1/M_PI_2;}
        vx=dgrid*a*vx;
        vy=-dgrid*a*vy;
        if( (vx<1.0/xscale)&&(vx>-1.0/xscale)&&(vy<1.0/yscale)&&(vy>-1.0/yscale) )
          {vx=1.0/xscale;}
        grid_data[j*2*grid_num+2*i]=vx;
        grid_data[j*2*grid_num+2*i+1]=vy;
       }
 return;
}

/***********************************/
/*sets the time frame to draw until*/
/***********************************/
void phaseplane_set_time_frame(double tmin, double tmax)
{
 draw_tmin=MINIMUM(tmin, 0.0);
 draw_tmax=MAXIMUM(tmax, 0.0);
 return;
}

/***********************************/
/*gets the time frame to draw until*/
/***********************************/
void phaseplane_get_time_frame(double *tmin, double *tmax)
{
 *tmin=draw_tmin;
 *tmax=draw_tmax;
 return;
}

/************************************************/
/*returns: 1, if any trajectory has changed set,*/
/*         0 otherwise                          */
/************************************************/
int phaseplane_trajectory_changed()
{trajectory_t *ft;
 
 ft=trajectory;
 while(ft)
    {if(ft->changed!=0)
       {return 1;}
     ft=ft->next;
    }
 return 0;
}

/************************/
/*draws a directionfield*/
/************************/
void phaseplane_draw_dirfield(cairo_t *cr)
{double vx, vy, dgridx, dgridy, nx, ny, ex, ey;
 int i,j;
 
 if(grid_num<=0){return;}
 if(cr==NULL){return;}
 /*calculate the distance of grid points*/
 dgridx=(pp_xmax-pp_xmin)/((double)grid_num);
 dgridy=(pp_ymax-pp_ymin)/((double)grid_num);
 /*drawing the grid points*/
 for(i=0;i<grid_num;i++)
    for(j=0;j<grid_num;j++)
       {/*drawing an arrow*/
        ex=PP2CR_X(pp_xmin+dgridx*((double)i+0.5));
        ey=PP2CR_Y(pp_ymin+dgridy*((double)j+0.5));
        vx=xscale*grid_data[j*2*grid_num+2*i];
        vy=yscale*grid_data[j*2*grid_num+2*i+1];
        cairo_move_to(cr, ex, ey);
        cairo_line_to(cr, ex+0.75*vx, ey+0.75*vy);
        ex+=vx; ey+=vy;
        cairo_stroke(cr);
        cairo_move_to(cr, ex, ey);
        vx*=0.4; vy*=0.4;
        nx=0.866*vx-0.5*vy;
        ny=0.5*vx+0.866*vy;
        cairo_line_to(cr, ex-nx, ey-ny);
        nx+=vy;
        ny-=vx;
        cairo_line_to(cr, ex-nx, ey-ny);
        cairo_line_to(cr, ex, ey);
        cairo_close_path(cr);
        cairo_fill(cr);
        cairo_stroke(cr);
       }
 return;
}

/************************************/
/*redraws the whole phaseplane to cr*/
/************************************/
void phaseplane_draw(cairo_t *cr, double width, double height)
{double d;
 trajectory_t *ft;

 /*set drawing square*/
 if(cr==NULL){return;}
 if( (width<=0.0)||(height<=0.0) ){return;}
 d=MINIMUM(width,height)-1.0;
 cr_ymax=height-1.0;
 xscale=d/(pp_xmax-pp_xmin);
 yscale=d/(pp_ymax-pp_ymin);
 yoffset=-yscale*pp_ymin+cr_ymax-d;
 d+=1.0;
 xoffset=-xscale*pp_xmin+(width-d)*0.5;
 
 /*bounding box and axis*/
 cairo_rectangle(cr,0.5*(width-d),0,d,d);
 cairo_clip_preserve(cr);
 /*cairo_set_antialias(cr, CAIRO_ANTIALIAS_NONE);*/
 cairo_set_source_rgb (cr, rbackgr, gbackgr, bbackgr);
 cairo_fill(cr);
 cairo_set_source_rgb (cr, raxis, gaxis, baxis);
 cairo_set_line_join(cr,CAIRO_LINE_JOIN_MITER);
 cairo_set_line_cap(cr,CAIRO_LINE_CAP_BUTT);
 cairo_set_line_width(cr,2.0);
 cairo_rectangle(cr,0.5*(width-d),0,d-0.5,d-0.5);
 cairo_set_line_width(cr,1.0);
 /*y axe*/
 cairo_move_to(cr, PP2CR_X(0.0), PP2CR_Y(pp_ymin));
 cairo_line_to(cr, PP2CR_X(0.0), PP2CR_Y(pp_ymax));
 /*x axe*/
 cairo_move_to(cr, PP2CR_X(pp_xmin), PP2CR_Y(0.0));
 cairo_line_to(cr, PP2CR_X(pp_xmax), PP2CR_Y(0.0));
 cairo_stroke(cr);

 /*direction field*/
 cairo_set_source_rgb (cr, rdirf, gdirf, bdirf);
 phaseplane_draw_dirfield(cr);
 /*trajectories*/
 ft=trajectory;
 while(ft)
    {trajectory_draw_poi(ft, cr, 0.0, 0.0);
     ft=ft->next;
    }
 return;
}

/**********************************************/
/*draws trajectories from startplus/startminus*/
/**********************************************/
void phaseplane_draw_delta(cairo_t *cr, double width, double height,
                           double startplus, double startminus)
{double d;
 trajectory_t *ft;

 /*set drawing square*/
 if(cr==NULL){return;}
 if( (width<=0.0)||(height<=0.0) ){return;}
 d=MINIMUM(width,height)-1.0;
 cr_ymax=height-1.0;
 xscale=d/(pp_xmax-pp_xmin);
 yscale=d/(pp_ymax-pp_ymin);
 yoffset=-yscale*pp_ymin+cr_ymax-d;
 d+=1.0;
 xoffset=-xscale*pp_xmin+(width-d)*0.5;
 /*clipping*/
 cairo_rectangle(cr,0.5*(width-d),0,d,d);
 cairo_clip(cr);
 /*trajectories*/
 ft=trajectory;
 while(ft)
    {trajectory_draw_poi(ft, cr, startplus, startminus);
     ft=ft->next;
    }
 return;
}

/***************************************/
/*draws dots to the end of trajectories*/
/***************************************/
void phaseplane_draw_dots(cairo_t *cr, double width, double height,
                          double sizeplus, double sizeminus)
{double d;
 trajectory_t *ft;

 /*set drawing square*/
 if(cr==NULL){return;}
 if( (width<=0.0)||(height<=0.0) ){return;}
 d=MINIMUM(width,height)-1.0;
 cr_ymax=height-1.0;
 xscale=d/(pp_xmax-pp_xmin);
 yscale=d/(pp_ymax-pp_ymin);
 yoffset=-yscale*pp_ymin+cr_ymax-d;
 d+=1.0;
 xoffset=-xscale*pp_xmin+(width-d)*0.5;
 /*clipping*/
 cairo_rectangle(cr,0.5*(width-d),0,d,d);
 cairo_clip(cr);
 /*dots*/
 ft=trajectory;
 while(ft)
    {trajectory_dot_draw_poi(ft, cr, sizeplus, sizeminus);
     ft=ft->next;
    }
 return;
}

/**************************************/
/*sets the variables shown on the axis*/
/*-1 indicates time, 0 is the 1st var.*/
/**************************************/
void phaseplane_set_shown_variables(int newix, int newiy)
{
 if( (newix>=-1)&&(newix<o. dimension) ){ix=newix;}
 if( (newiy>=-1)&&(newiy<o. dimension) ){iy=newiy;}
 return;
}

/***********************************/
/*Saves the current ODE, parameters*/
/*and trajectories to a text file. */
/*return: 0 on success, -1 on error*/
/***********************************/
int phaseplane_save_to_file(FILE *outf)
{int i;
 char str[4096];
 trajectory_t *ft;

 if(outf==NULL){return -1;}
 fprintf(outf,"#Saved Phasepictor phaseplane\n#Equations section\n#Number of variables\n");
 fprintf(outf,"%d\n", o.dimension);
 fprintf(outf,"#Names of the variables and the equations for their derivatives\n");
 for(i=0;i<o.dimension;i++)
    {kiir(o.equation[i], str);
     fprintf(outf,"%s  %s\n", valtozok[2+i].nev, str);
    }
 fprintf(outf,"#Number of parameters\n");
 fprintf(outf,"%d\n", o.nparam);
 fprintf(outf,"#Parameter name-value pairs\n");
 for(i=0;i<o.nparam;i++)
    {fprintf(outf,"%s %g\n", o.parameter[i].name, o.parameter[i].value);
    }
 fprintf(outf,"#Number of trajectories\n");
 fprintf(outf,"%d\n",numtraj);
 fprintf(outf,"#Trajectories data\n#Each line contains one trajectory: tmin, tmax, dt, initconds\n");
 ft=trajectory;
 while(ft)
    {fprintf(outf,"%g %g %g", ft->tmin, ft->tmax, ft->dt);
     for(i=0;i<o.dimension;i++)
        {fprintf(outf," %g", ft->initcond[i]);}
     fprintf(outf,"\n");
     ft=ft->next;
    }
 fprintf(outf,"#Global dt (0.0 means no global dt)\n");
 fprintf(outf,"%g\n", o.global_dt);
 fprintf(outf,"#Phaseplane's data: xmin, xmax, ymin, ymax, gridsize of direction field\n");
 fprintf(outf,"%g %g %g %g %d\n",pp_xmin, pp_xmax, pp_ymin, pp_ymax, grid_num);
 return 0;
}
