/*
  PhasePictor - A program for displaying phaseplane of an ODE
  Copyright (c) 2007-2009 Kristóf Kály-Kullai
  This program is free software, it is distributed under the terms 
  of the GNU General Public License version 3 or (at your option) later.
  For details see the attached GPL-3.txt file.
  The software is provided "AS IS", WITHOUT WARRANTY of any kind.
*/
#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <cairo/cairo-svg.h>
#include <glade/glade.h>
#include <string.h>
#include <locale.h>
#include "phasepictor_numeric.h"
#include "kparser.h"

/**********************************/
/* G L O B A L  V A R I A B L E S */
/**********************************/

#if GTK_MINOR_VERSION>=10
GtkPrintSettings *PrintSettings;
#endif
GtkWidget *dialogError;
GtkWidget *dialogOpen, *dialogSave, *dialogExport;
gchar *filenameSave, *folderExport;
GtkFileFilter *allFilter, *pngFilter, *svgFilter, *pplFilter;
GtkWidget *mainWindow;
GtkWidget *stbMain;
GtkWidget *menuBar;
guint stb_context;
GtkWidget *expPhasePlane, *expEquation, *expParameter, *expTrajectory;
GtkWidget *draPhasePlane;
cairo_surface_t *csPhasePlane=NULL;
cairo_t *crPhasePlane=NULL;
GtkWidget *tbFullscreen;
GtkWidget *sbXmin;
GtkWidget *sbXmax;
GtkWidget *sbYmin;
GtkWidget *sbYmax;
GtkWidget *sbGridsize;
GtkWidget *chbPrintColors;
GtkWidget *entXdot;
GtkWidget *entYdot;
GtkWidget *butSetEquation;
GtkWidget *cmbParameter;
GtkWidget *cmbentParameter;
GtkWidget *sbParam;
GtkWidget *cmbTrajectory;
GtkWidget *draTrajectory;
GtkWidget *sbX0;
GtkWidget *sbY0;
GtkWidget *sbTmin;
GtkWidget *sbTmax;
GtkWidget *sbDt;
GtkWidget *chbSameDt;
GtkWidget *butCalculate, *butCalculateAll, *butStop;
GtkWidget *tbPrint;
GtkWidget *expAnimation;
GtkWidget *sbTunit;
GtkWidget *butAnimate;
gint iParam;
gboolean editParam, editTraj, editXdot, editYdot, SameDt;
gchar messageStr[512], stbStr[128];
int ode_dimension=2, ode_nparam=4, ntraj=0, cumntraj=0;
int stop=0;
#define ANIM_MIN_TUNIT 0.001
#define ANIM_MAX_TUNIT 1000
#define ANIM_FPS 10.0
/*Flash_Freq: in 1/ANIM_FPS units the time spent in one state (on or off)*/
int Flash_Freq=0;
int Anim_Type=0;
#define ANIM_DOTSIZE 10.0
double animDt=0.1, maxtmax=0.0, mintmin=0.0;
double current_plus_dotsize=0.0, current_minus_dotsize=0.0;
int animplus=0, animminus=0, animplusi=0, animminusi=0;
/*New model dialog*/
GtkWidget *dlgNewModel;
GtkWidget *sbNumVariables;
GtkWidget *sbNumParameters;

gchar par_name[16];

/*********************/
/* F U N C T I O N S */
/*********************/


/*********************/
/* C A L L B A C K S */
/*********************/

void message(const gchar *mess)
{
 GtkWidget *message;

 message=gtk_message_dialog_new(GTK_WINDOW(mainWindow),
                                GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
                                GTK_MESSAGE_INFO,
                                GTK_BUTTONS_CLOSE,
                                mess);
 gtk_window_set_title(GTK_WINDOW(message), "Message");
 gtk_dialog_run(GTK_DIALOG(message));
 gtk_widget_destroy(message);
}

void error_message(const gchar *emes)
{
 dialogError=gtk_message_dialog_new(GTK_WINDOW(mainWindow),
                                    GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_ERROR,
                                    GTK_BUTTONS_CLOSE,
                                    emes);
 gtk_window_set_title(GTK_WINDOW(dialogError), "Error");
 g_signal_connect(dialogError, "response", G_CALLBACK(gtk_widget_destroy), NULL);
 gtk_widget_show(dialogError);
}

/*on delete-event of mainWindow*/
G_MODULE_EXPORT gint cbExit(GtkWidget *widget,GdkEvent *event,gpointer data )
{
 GtkWidget *dialog;
 
 dialog=gtk_message_dialog_new(GTK_WINDOW(mainWindow),
                               GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
                               GTK_MESSAGE_QUESTION,
                               GTK_BUTTONS_YES_NO,
                               (const gchar *)"Are you sure you want to quit?");
 if( gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_YES )
   {gtk_widget_destroy(dialog);
	gtk_main_quit();
	return FALSE;
   }
 else
   {gtk_widget_destroy(dialog);
	return TRUE;
   }
} 

/*on clicked dlgNewModel/butNewCreate*/
G_MODULE_EXPORT void cbNewCreate (GtkButton *button, gpointer user_data)
{gchar str[128];
 if( gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbNumVariables))+gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbNumParameters))>MAX_VALT-2 )
   {g_sprintf(str,"The total number of variables\nand parameters exceeded %d.",MAX_VALT-2);
    error_message(str);
   }
 else
   {gtk_dialog_response(GTK_DIALOG(dlgNewModel), GTK_RESPONSE_ACCEPT);}
}

/*on clicked dlgNewModel/butNewCancel*/
G_MODULE_EXPORT void cbNewCancel (GtkButton *button, gpointer user_data)
{
 gtk_dialog_response(GTK_DIALOG(dlgNewModel), GTK_RESPONSE_CANCEL);
}

/*on clicked tbNew and on activate menuNew*/
G_MODULE_EXPORT void cbNew (GtkMenuItem *menuitem, gpointer user_data)
{int i;
 double d1, d2, d3, d4;

 if( gtk_dialog_run(GTK_DIALOG(dlgNewModel))==GTK_RESPONSE_ACCEPT )
   {/*destroy the old model*/
    /*reset Equation @later with n dimensions*/
    gtk_widget_set_sensitive(butSetEquation,FALSE);
    /*reset Parameter*/
    for(i=0;i<ode_nparam;i++)
       {gtk_combo_box_remove_text(GTK_COMBO_BOX(cmbParameter), 0);}
    gtk_entry_set_text(GTK_ENTRY(cmbentParameter), "");
    /*reset Trajectory*/
    for(i=0;i<ntraj;i++)
       {gtk_combo_box_remove_text(GTK_COMBO_BOX(cmbTrajectory), 0);}
    gtk_widget_set_sensitive(sbX0,FALSE);
    gtk_widget_set_sensitive(sbY0,FALSE);
    gtk_widget_set_sensitive(sbTmin,FALSE);
    gtk_widget_set_sensitive(sbTmax,FALSE);
    gtk_widget_set_sensitive(sbDt,FALSE);
    gtk_widget_set_sensitive(butCalculate,FALSE);
    gtk_widget_set_sensitive(butCalculateAll,FALSE);
    ntraj=0; cumntraj=0;
    phaseplane_destroy();

    /*create a new model*/
    ode_dimension=(int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbNumVariables));
    ode_nparam=(int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbNumParameters));
    
    if( phaseplane_init(ode_dimension,ode_nparam) )
      {g_error("Unable to initialize phaseplane.\n");}
    /*Set new parameters*/
    for(i=0;i<MINIMUM(ode_nparam,4);i++)
       {g_sprintf(par_name,"%c",i+97);
        ode_set_parameter_name(i,par_name);
        ode_set_parameter_value(i,1.0);
        gtk_combo_box_append_text( GTK_COMBO_BOX(cmbParameter), par_name );
       }
    for(;i<ode_nparam;i++)
       {g_sprintf(par_name,"p%d",i);
        ode_set_parameter_name(i,par_name);
        ode_set_parameter_value(i,1.0);
        gtk_combo_box_append_text( GTK_COMBO_BOX(cmbParameter), par_name );
       }
    gtk_combo_box_set_active( GTK_COMBO_BOX(cmbParameter), 0);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbParam), 1.0);
    /*set new variables*/
    ode_set_variable_name(0, "x"); ode_set_variable_name(1, "y");
    gtk_entry_set_text(GTK_ENTRY(entXdot), "0");
    gtk_entry_set_text(GTK_ENTRY(entYdot), "0");
    ode_set_equation(0, gtk_editable_get_chars( GTK_EDITABLE(entXdot), 0, -1) );
    ode_set_equation(1, gtk_editable_get_chars( GTK_EDITABLE(entYdot), 0, -1) );
    gtk_widget_set_sensitive(butSetEquation,FALSE);
    /*set Trajectory*/
    if(SameDt)
      {ode_set_global_dt(0.01);
       gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbDt), (gdouble)ode_get_global_dt());
      }
    else
      {ode_set_global_dt(0.0);
      }
    /*set the Phaseplane values*/
    phaseplane_get_window(&d1, &d2, &d3, &d4);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbXmin), d1);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbXmax), d2);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbYmin), d3);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbYmax), d4);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbGridsize), (gdouble)phaseplane_get_dirfield_grid() );
    /*invoke redraw*/
    gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
   }
 gtk_widget_hide(GTK_WIDGET(dlgNewModel));
}

#define LINE_LENGTH 4096
/*on clicked tbOpen and on activate menuOpen*/
G_MODULE_EXPORT void cbOpen (GtkWidget *menuitem, gpointer user_data)
{char str[LINE_LENGTH], str2[32];
 gchar *filename;
 FILE *input;
 int i, j, dim, par, tra, grn;
 double d, gdt, xmi, xma, ymi, yma;
 
 if( gtk_dialog_run(GTK_DIALOG(dialogOpen)) == GTK_RESPONSE_ACCEPT )
   {/*read from file*/
    filename=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialogOpen));
    input=g_fopen(filename,"rt");
    if(input==NULL){error_message("Unable to open file.");}

    /*check if the file is valid and without errors*/
    setlocale(LC_ALL,"C");
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    if(pars_string(str, LINE_LENGTH)!=1){goto FINITO;}
    if(sscanf(get_token(0), "%d", &dim)!=1){goto FINITO;}
    if(dim<1)
      {gtk_widget_hide(dialogOpen);
       setlocale(LC_ALL,"");
       error_message("Number of dimensions is less than one.");
       return;
      } 
    for(i=0;i<dim;i++)
       {if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
        if(pars_string(str, LINE_LENGTH)!=2){goto FINITO;}
        /*return if variable name is a number*/
        if(sscanf(get_token(0),"%lg",&d)==1){goto FINITO;}
       }
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    if(pars_string(str, LINE_LENGTH)!=1){goto FINITO;}
    if(sscanf(get_token(0), "%d", &par)!=1){goto FINITO;}
    if(par<0)
      {gtk_widget_hide(dialogOpen);
       setlocale(LC_ALL,"");
       error_message("Number of parameters is negative.");
       return;
      }
    if(dim+par>MAX_VALT-2)
      {gtk_widget_hide(dialogOpen);
       g_sprintf(str,"The total number of variables and parameters exceeded %d.",MAX_VALT-2);
       setlocale(LC_ALL,"");
       error_message(str);
       return;
      }
    for(i=0;i<par;i++)
       {if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
        if(pars_string(str, LINE_LENGTH)!=2){goto FINITO;}
        /*return if parameter name is a number*/
        if(sscanf(get_token(0),"%lg",&d)==1){goto FINITO;}
        /*return if parameter value is not a number*/
        if(sscanf(get_token(1),"%lg",&d)!=1){goto FINITO;}
       }
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    if(pars_string(str, LINE_LENGTH)!=1){goto FINITO;}
    if(sscanf(get_token(0), "%d", &tra)!=1){goto FINITO;}
    if(tra<0)
      {gtk_widget_hide(dialogOpen);
       setlocale(LC_ALL,"");
       error_message("Number of trajectories is negative.");
       return;
      }
    for(i=0;i<tra;i++)
       {if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
        if(pars_string(str, LINE_LENGTH)!=3+dim){goto FINITO;}
        for(j=0;j<3+dim;j++)
           {if(sscanf(get_token(j), "%lg", &d)!=1){goto FINITO;}
           }
       }
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    if(pars_string(str, LINE_LENGTH)!=1){goto FINITO;}
    if(sscanf(get_token(0), "%lg", &gdt)!=1){goto FINITO;}
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    if(pars_string(str, LINE_LENGTH)!=5){goto FINITO;}
     if(sscanf(get_token(0), "%lg", &xmi)!=1){goto FINITO;}
     if(sscanf(get_token(1), "%lg", &xma)!=1){goto FINITO;}
     if(sscanf(get_token(2), "%lg", &ymi)!=1){goto FINITO;}
     if(sscanf(get_token(3), "%lg", &yma)!=1){goto FINITO;}
     if(sscanf(get_token(4), "%d", &grn)!=1){goto FINITO;}
    
    /*everything OK, so destroy the old model*/
    /*reset Equation @later with n dimensions*/
    setlocale(LC_ALL,"");
    gtk_widget_set_sensitive(butSetEquation,FALSE);
    /*reset Parameter*/
    for(i=0;i<ode_nparam;i++)
       {gtk_combo_box_remove_text(GTK_COMBO_BOX(cmbParameter), 0);}
    gtk_entry_set_text(GTK_ENTRY(cmbentParameter), "");
    /*reset Trajectory*/
    for(i=0;i<ntraj;i++)
       {gtk_combo_box_remove_text(GTK_COMBO_BOX(cmbTrajectory), 0);}
    gtk_widget_set_sensitive(sbX0,FALSE);
    gtk_widget_set_sensitive(sbY0,FALSE);
    gtk_widget_set_sensitive(sbTmin,FALSE);
    gtk_widget_set_sensitive(sbTmax,FALSE);
    gtk_widget_set_sensitive(sbDt,FALSE);
    gtk_widget_set_sensitive(butCalculate,FALSE);
    gtk_widget_set_sensitive(butCalculateAll,FALSE);
    ntraj=0; cumntraj=0;
    phaseplane_destroy();
    /*@remove, when n dimensions are supported*/
    if(dim!=2)
      {gtk_widget_hide(dialogOpen);
       message("Only 2 dimensions are supported yet.");
       return;
      }
    
    /*now creating a new model by reading from file*/
    fseek(input, 0, SEEK_SET);
    /*create a new model*/
    ode_dimension=dim;
    ode_nparam=par;
    if( phaseplane_init(ode_dimension,ode_nparam) )
      {g_error("Unable to initialize phaseplane.\n");}
    /*skip number of variables, it is already in dim*/
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    /*set up variables*/
    for(i=0;i<dim;i++)
       {if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
        if(pars_string(str, LINE_LENGTH)!=2){goto FINITO;}
        ode_set_variable_name(i, get_token(0));
        /*@change with n diemnsions*/
        switch(i)
          {case 0:{gtk_entry_set_text(GTK_ENTRY(entXdot), get_token(1));
                   break;}
           case 1:{gtk_entry_set_text(GTK_ENTRY(entYdot), get_token(1));
                   break;}
          }
       }
    /*set up parameters*/
    /*skip number of parameters, it is already in par*/
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    for(i=0;i<par;i++)
       {if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
        if(pars_string(str, LINE_LENGTH)!=2){goto FINITO;}
        ode_set_parameter_name(i, get_token(0));
        gtk_combo_box_append_text( GTK_COMBO_BOX(cmbParameter), get_token(0) );
        setlocale(LC_ALL,"C");
        if(sscanf(get_token(1),"%lg",&d)!=1)
          {setlocale(LC_ALL,"");
           error_message("Parameter value not a number, set to default (0).");
           d=0.0;
          }
        setlocale(LC_ALL,"");
        ode_set_parameter_value(i,d);
       }
    gtk_combo_box_set_active( GTK_COMBO_BOX(cmbParameter), 0);
    /*now set the equations*/
    /*@change with n diemnsions*/
    j=ode_set_equation(0, gtk_editable_get_chars( GTK_EDITABLE(entXdot), 0, -1) );
    if(j!=0)
      {if(j==-1)
         {ode_set_equation(0, "0");
          error_message("Equation for x' has bad format, set to default (0).");
         }
       else
         {error_message("Equation for x' has bad format, was just partially read.");}
      }
    ode_get_equation(0, str);
    gtk_entry_set_text(GTK_ENTRY(entXdot), str);
    j=ode_set_equation(1, gtk_editable_get_chars( GTK_EDITABLE(entYdot), 0, -1) );
    if(j!=0)
      {if(j==-1)
         {ode_set_equation(1, "0");
          error_message("Equation for y' has bad format, set to default (0).");
         }
       else{error_message("Equation for y' has bad format, was just partially read.");}
      }
    ode_get_equation(1, str);
    gtk_entry_set_text(GTK_ENTRY(entYdot), str);
    gtk_widget_set_sensitive(butSetEquation,FALSE);
    /*set up trajectories*/
    /*skip number of trajectories, it is already in tra*/
    if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
    ode_set_global_dt(gdt);
    if( ode_get_global_dt() > 0.0 )
      {gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chbSameDt), TRUE);
       gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbDt), ode_get_global_dt() );
      }
    else
      {gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chbSameDt), FALSE);}
    for(i=0;i<tra;i++)
       {if(nextline(input, str, LINE_LENGTH)==-1){goto FINITO;}
        if(pars_string(str, LINE_LENGTH)!=3+dim){goto FINITO;}
        g_sprintf(str2, "traj%d", cumntraj);
        if( trajectory_add(str2,cumntraj) )
          {error_message("Error: unable to add new trajectory.");}
        else
          {cumntraj++;
           gtk_combo_box_append_text( GTK_COMBO_BOX(cmbTrajectory), str2 );
           ntraj++;
           setlocale(LC_ALL,"C");
           if(sscanf(get_token(0), "%lg", &d)!=1)
             {d=0.0;
              setlocale(LC_ALL,"");
              error_message("Tmin value not a number, set to default (0).");
             }
           trajectory_set_tmin(str2, d);
           if(sscanf(get_token(1), "%lg", &d)!=1)
             {d=5.0;
              setlocale(LC_ALL,"");
              error_message("Tmax value not a number, set to default (5).");
             }
           trajectory_set_tmax(str2, d);
           if(sscanf(get_token(2), "%lg", &d)!=1)
             {d=0.01;
              setlocale(LC_ALL,"");
              error_message("Deltat value not a number, set to default (0.01).");
             }
           trajectory_set_dt(str2, d);
           /*@rewrite with n dimensions*/
           if(sscanf(get_token(3), "%lg", &d)!=1)
             {d=0.0;
              setlocale(LC_ALL,"");
              error_message("X0 value not a number, set to default (0).");
             }
           trajectory_set_initcond(str2, 0, d);
           if(sscanf(get_token(4), "%lg", &d)!=1)
             {d=0.0;
              setlocale(LC_ALL,"");
              error_message("Y0 value not a number, set to default (0).");
             }
           trajectory_set_initcond(str2, 1, d);
           setlocale(LC_ALL,"");
          }
       }
    if(tra)
      {gtk_widget_set_sensitive(sbX0,TRUE);
       gtk_widget_set_sensitive(sbY0,TRUE);
       gtk_widget_set_sensitive(sbTmin,TRUE);
       gtk_widget_set_sensitive(sbTmax,TRUE);
       gtk_widget_set_sensitive(sbDt,TRUE);
       gtk_widget_set_sensitive(butCalculate,TRUE);
       gtk_widget_set_sensitive(butCalculateAll,TRUE);
       gtk_combo_box_set_active(GTK_COMBO_BOX(cmbTrajectory), 0);
      }
    /*set up phaseplane*/
    phaseplane_set_window(xmi, xma, ymi, yma);
    phaseplane_set_dirfield_grid(grn);
    phaseplane_get_window(&xmi, &xma, &ymi, &yma);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbXmin), xmi);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbXmax), xma);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbYmin), ymi);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbYmax), yma);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbGridsize), 
                              (gdouble)phaseplane_get_dirfield_grid() );
    /*invoke redraw*/
    gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
    fclose(input);
    g_free(filename);
   }
 gtk_widget_hide(dialogOpen);
 return;
FINITO:
 g_free(filename);
 gtk_widget_hide(dialogOpen);
 error_message("Not a valid phaseplane file.");
 return;
}

/*on clicked tbSave and on activate menuSave*/
G_MODULE_EXPORT void cbSave (GtkWidget *menuitem, gpointer user_data)
{FILE *output;
 
/* pplFilter=gtk_file_filter_new();
 gtk_file_filter_set_name(pplFilter,(gchar *)"Saved phaseplane (*.ppl)");
 gtk_file_filter_add_pattern(pplFilter,(gchar *)"*.ppl");*/
 dialogSave = gtk_file_chooser_dialog_new ("Save phaseplane",
                                       GTK_WINDOW(mainWindow),
                                       GTK_FILE_CHOOSER_ACTION_SAVE,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                       GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                       (const gchar *)NULL);
 gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER(dialogSave), TRUE);
 gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialogSave), pplFilter);
 gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialogSave), allFilter);
 gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialogSave), pplFilter);
 gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialogSave), FALSE);
 if(filenameSave)
   {gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialogSave), filenameSave);}
 else
   {gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialogSave), "untitled.ppl");}

 if( gtk_dialog_run(GTK_DIALOG (dialogSave)) == GTK_RESPONSE_ACCEPT )
   {/*save to file*/
    filenameSave=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialogSave));
    output=g_fopen(filenameSave,"wt");
    if(output==NULL){error_message("Unable to open file.");}
    setlocale(LC_ALL,"C");
    phaseplane_save_to_file(output);
    fflush(output);
    setlocale(LC_ALL,"");
    fclose(output);
   }
/* gtk_object_destroy(GTK_OBJECT(pplFilter));*/
 gtk_widget_destroy(dialogSave);
}

/*on clicked tbSaveas and on activate menuSaveas*/
G_MODULE_EXPORT void cbSaveas (GtkWidget *menuitem, gpointer user_data)
{gchar *filenameExport, *filenameutf8Export, *filenameuncaseExport;
 cairo_surface_t *cs;
 cairo_t *cr;
 int is_png;
 gsize fnlen, utf8len;

 pngFilter=gtk_file_filter_new();
 gtk_file_filter_set_name(pngFilter,(gchar *)"PNG image (*.png)");
 gtk_file_filter_add_pattern(pngFilter,(gchar *)"*.png");
#if CAIRO_HAS_SVG_SURFACE
 svgFilter=gtk_file_filter_new();
 gtk_file_filter_set_name(svgFilter,(gchar *)"SVG image (*.svg)");
 gtk_file_filter_add_pattern(svgFilter,(gchar *)"*.svg");
#endif
 dialogExport = gtk_file_chooser_dialog_new ("Export to image",
                                       GTK_WINDOW(mainWindow),
                                       GTK_FILE_CHOOSER_ACTION_SAVE,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                       GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                       (const gchar *)NULL);
 gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER(dialogExport), TRUE);
 gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialogExport), pngFilter);
#if CAIRO_HAS_SVG_SURFACE
 gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialogExport), svgFilter);
#endif
 gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialogExport), allFilter);
 gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialogExport), pngFilter);
 gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialogExport), FALSE);
 if(folderExport)
   {gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialogExport), folderExport);}
 gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialogExport), "untitled.png"); 

 if( gtk_dialog_run(GTK_DIALOG(dialogExport)) == GTK_RESPONSE_ACCEPT )
   {/*exporting the phaseplane to png*/
    folderExport=gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(dialogExport));
    filenameExport=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialogExport));
    filenameutf8Export=g_filename_to_utf8(filenameExport, -1, &fnlen, &utf8len, NULL);
    g_free(filenameExport);
    if(filenameutf8Export==NULL)
      {error_message("Unable to convert filename to UTF8.");
       g_free(filenameutf8Export);
       gtk_widget_hide(dialogExport);
       return;
      }
#if CAIRO_HAS_SVG_SURFACE
    filenameuncaseExport=g_utf8_casefold(filenameutf8Export, -1);
    if( g_utf8_collate(filenameuncaseExport+strlen(filenameuncaseExport)-3, "svg")==0 )
      {is_png=0;}
    else{is_png=1;}
    g_free(filenameuncaseExport);
#endif
    filenameExport=g_locale_from_utf8(filenameutf8Export, -1, &utf8len, &fnlen, NULL);
    g_free(filenameutf8Export);
    if(filenameExport==NULL)
      {error_message("Unable to convert filename to the C locale.");
       g_free(filenameExport);
       gtk_widget_hide(dialogExport);
       return;
      }
    /*initialize cairo*/
#if CAIRO_HAS_SVG_SURFACE
    if(is_png)
      {
#endif
       cs=cairo_image_surface_create(CAIRO_FORMAT_RGB24,512,512);
       if(cs==NULL)
         {error_message("Error exporting as PNG image.");
          g_free(filenameExport);
          gtk_widget_hide(dialogExport);
          return;
         }
#if CAIRO_HAS_SVG_SURFACE
      }
    else
      {cs=cairo_svg_surface_create(filenameExport, 512.0, 512.0);
       if(cs==NULL)
         {error_message("Error exporting as SVG image.");
          g_free(filenameExport);
          gtk_widget_hide(dialogExport);
          return;
         }
      }
#endif
    cr=cairo_create(cs);
    if(cairo_status(cr)!=CAIRO_STATUS_SUCCESS)
      {error_message("Error while initializing Cairo.");
       cairo_destroy(cr);
       cairo_surface_destroy(cs);
       g_free(filenameExport);
       gtk_widget_hide(dialogExport);
       return;
      }
    cairo_rectangle(cr, 0.0, 0.0, 512.0, 512.0);
    cairo_clip(cr);
    
    /*drawing to cairo*/
    phaseplane_set_trajectory_color(0.0, 0.0, 0.0);
    phaseplane_set_draw_color_traj(
               gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chbPrintColors)));
    phaseplane_set_background_color(1.0, 1.0, 1.0);
    phaseplane_set_axis_color(0.1, 0.1, 0.1);
    phaseplane_set_dirfield_color(0.1, 0.1, 0.1);
    phaseplane_draw(cr, 512.0, 512.0);
    
    cairo_show_page(cr);
    cairo_surface_flush(cs);
    /*saving cairo to png*/
#if CAIRO_HAS_SVG_SURFACE
    if(is_png)
      {
#endif
       if( cairo_surface_write_to_png(cs, filenameExport)!=0 )
         {error_message("Error writing to .png file.");
         }
#if CAIRO_HAS_SVG_SURFACE
      }
#endif
      
    /*finishing*/
    cairo_destroy(cr);
    cairo_surface_destroy(cs);
    g_free(filenameExport);
   }
 
 gtk_object_destroy(GTK_OBJECT(pngFilter));
#if CAIRO_HAS_SVG_SURFACE
 gtk_object_destroy(GTK_OBJECT(svgFilter));
#endif
 gtk_widget_destroy(dialogExport);
}

/*on draw_page of a GtkPrintOperation*/
#if GTK_MINOR_VERSION>=10
G_MODULE_EXPORT void cbDrawPage (GtkPrintOperation *operation, GtkPrintContext *context, gint page_nr, gpointer user_data)
{cairo_t *cr;
 
 /*setting up cairo_t*/
 cr=gtk_print_context_get_cairo_context(context);
 
 /*drawing*/
 phaseplane_set_trajectory_color(0.0, 0.0, 0.0);
 phaseplane_set_draw_color_traj(
            gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chbPrintColors)));
 phaseplane_set_background_color(1.0, 1.0, 1.0);
 phaseplane_set_axis_color(0.1, 0.1, 0.1);
 phaseplane_set_dirfield_color(0.1, 0.1, 0.1);
 phaseplane_draw(cr, gtk_print_context_get_width(context), gtk_print_context_get_height(context));
}
#endif

/*on clicked tbPrint and on activate menuPrint*/
G_MODULE_EXPORT void cbPrint (GtkWidget *menuitem, gpointer user_data)
{
 #if GTK_MINOR_VERSION>=10
 GtkPrintOperation *print;
 GtkPrintOperationResult res;
 GError *error;

 /*setting up print operation*/
 print=gtk_print_operation_new();
 if( PrintSettings!=NULL )
   {gtk_print_operation_set_print_settings(print, PrintSettings);}
 gtk_print_operation_set_n_pages(print, 1);
 g_signal_connect(print, "draw_page", G_CALLBACK (cbDrawPage), NULL);
 /*printing*/
 res=gtk_print_operation_run(print, GTK_PRINT_OPERATION_ACTION_PRINT_DIALOG, GTK_WINDOW(mainWindow), &error);
 /*error handling and saving settings*/
 if( res == GTK_PRINT_OPERATION_RESULT_ERROR )
   {dialogError=gtk_message_dialog_new(GTK_WINDOW(mainWindow),
  			                     GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
					     GTK_MESSAGE_ERROR,
					     GTK_BUTTONS_CLOSE,
					     "Error printing file:\n%s",
					     (const gchar *)error->message);
    g_signal_connect(dialogError, "response", G_CALLBACK(gtk_widget_destroy), NULL);
    gtk_widget_show(dialogError);
    g_error_free(error);
   }
 else if( res == GTK_PRINT_OPERATION_RESULT_APPLY )
   {if( PrintSettings != NULL )
    g_object_unref(PrintSettings);
    PrintSettings=g_object_ref(gtk_print_operation_get_print_settings(print));
   }
 g_object_unref (print);
 #endif
 return ;
}

/*on toggled tbFullscreen*/
G_MODULE_EXPORT void cbFullscreen (GtkWidget *menuitem, gpointer user_data)
{
 if( gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(tbFullscreen)) )
   {gtk_widget_hide(GTK_WIDGET(expPhasePlane));
    gtk_widget_hide(GTK_WIDGET(stbMain));
    gtk_widget_hide(GTK_WIDGET(menuBar));
    gtk_widget_hide(GTK_WIDGET(expEquation));
    gtk_widget_hide(GTK_WIDGET(expParameter));
    gtk_widget_hide(GTK_WIDGET(expTrajectory));
    gtk_widget_hide(GTK_WIDGET(expAnimation));
    gtk_widget_hide(GTK_WIDGET(butCalculateAll));
    gtk_widget_hide(GTK_WIDGET(butStop));
   }
 else
   {gtk_widget_show(GTK_WIDGET(expPhasePlane));
    gtk_widget_show(GTK_WIDGET(stbMain));
    gtk_widget_show(GTK_WIDGET(menuBar));
    gtk_widget_show(GTK_WIDGET(expEquation));
    gtk_widget_show(GTK_WIDGET(expParameter));
    gtk_widget_show(GTK_WIDGET(expTrajectory));
    gtk_widget_show(GTK_WIDGET(expAnimation));
    gtk_widget_show(GTK_WIDGET(butCalculateAll));
    gtk_widget_show(GTK_WIDGET(butStop));
   }
}

/*on clicked tbQuit and on activate menuQuit*/
G_MODULE_EXPORT void cbQuit (GtkWidget *menuitem, gpointer user_data)
{GtkWidget *dialog;
 
 dialog=gtk_message_dialog_new(GTK_WINDOW(mainWindow),
                               GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
                               GTK_MESSAGE_QUESTION,
                               GTK_BUTTONS_YES_NO,
                               (const gchar *)"Are you sure you want to quit?");
 if( gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_YES )
   {gtk_main_quit ();}   
 gtk_widget_destroy(dialog);
}

/*on activate menuAbout*/
G_MODULE_EXPORT void cbAbout (GtkMenuItem *menuitem, gpointer user_data)
{gchar *strAuthors[]={"Kristóf Kály-Kullai", NULL};

 gtk_show_about_dialog(GTK_WINDOW(mainWindow),
                       "authors", strAuthors,
                       "comments", "A program for displaying phaseplane of an ODE.",
                       "copyright", "Kristóf Kály-Kullai\n2007-2009",
                       "license", "This program is free software, it is distributed under the terms\n\
of the GNU General Public License version 3 or (at your option) later.\n\
For details see the attached GPL-3.txt file.\n\n\
The software is provided \"AS IS\", WITHOUT WARRANTY of any kind.\n\n\
This program uses the GTK and libglade libraries, which are licensed\n\
under GNU LGPL 2.1, for details see the attached LGPL-2.1.txt file.",
                       "name", "PhasePictor",
                       "version", "0.6.1",
                       (gchar*)NULL);
}

/*on expose-event draPhasePlane*/
G_MODULE_EXPORT gboolean cbDraw (GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{cairo_t *cr;
 double d;

  /*deal with resize in double-buffered mode*/
  if((csPhasePlane!=NULL)
     /*&&(event->type==GDK_SETTING)*/
     &&( MINIMUM((double)draPhasePlane->allocation.width,(double)draPhasePlane->allocation.height)
		 !=cairo_image_surface_get_width(csPhasePlane) ) )
    {cairo_destroy(crPhasePlane);
     crPhasePlane=NULL;
     cairo_surface_destroy(csPhasePlane);
     csPhasePlane=NULL;
     d=MINIMUM((double)draPhasePlane->allocation.width,(double)draPhasePlane->allocation.height);
     csPhasePlane=cairo_image_surface_create(CAIRO_FORMAT_RGB24, d, d);
     if(cairo_surface_status(csPhasePlane)!=CAIRO_STATUS_SUCCESS)
       {cairo_surface_destroy(csPhasePlane);
        csPhasePlane=NULL;
        crPhasePlane=NULL;
		g_warning("Unable to create resized cairo_image_surface, fall back to direct drawing.");
       }
     else
       {crPhasePlane=cairo_create(csPhasePlane);
        if(cairo_status(crPhasePlane)!=CAIRO_STATUS_SUCCESS)
          {cairo_destroy(crPhasePlane);
           crPhasePlane=NULL;
           cairo_surface_destroy(csPhasePlane);
           csPhasePlane=NULL;
		   g_warning("Unable to create resized cairo_image_surface, fall back to direct drawing.");
          }		
       }
     if(crPhasePlane)
       {cairo_rectangle(crPhasePlane, 0.0, 0.0, d, d);
        cairo_clip(crPhasePlane);
        phaseplane_set_trajectory_color(1.0, 1.0, 1.0);
        phaseplane_set_draw_color_traj(1);
        phaseplane_set_background_color(0.0, 0.0, 0.0);
        phaseplane_set_axis_color(0.8, 0.8, 0.8);
        phaseplane_set_dirfield_color(0.0, 1.0, 1.0); 
        phaseplane_draw(crPhasePlane, d, d);
       }
    }
  
  /*setting up cairo_t*/
  cr = gdk_cairo_create (widget->window);
     
  /*set clipping*/
  cairo_rectangle (cr, event->area.x, event->area.y,
                   event->area.width, event->area.height);
  cairo_clip (cr);
  
  /*drawing*/
  if(csPhasePlane==NULL)
    {phaseplane_set_trajectory_color(1.0, 1.0, 1.0);
     phaseplane_set_draw_color_traj(1);
     phaseplane_set_background_color(0.0, 0.0, 0.0);
     phaseplane_set_axis_color(0.8, 0.8, 0.8);
     phaseplane_set_dirfield_color(0.0, 1.0, 1.0);
     phaseplane_draw(cr, (double)widget->allocation.width, 
                     (double)widget->allocation.height);
     if((current_plus_dotsize>0.0)||(current_minus_dotsize>0.0))
       {phaseplane_draw_dots(cr, (double)widget->allocation.width,
                             (double)widget->allocation.height,
                             current_plus_dotsize, current_minus_dotsize);
       }
    }
  else
    {
     phaseplane_set_trajectory_color(1.0, 1.0, 1.0);
     phaseplane_set_draw_color_traj(1);
     d=MINIMUM((double)draPhasePlane->allocation.width,(double)draPhasePlane->allocation.height);
     cairo_rectangle(cr,0.5*((double)draPhasePlane->allocation.width-d),0,d,d);
     cairo_clip(cr);
     cairo_set_source_surface(cr, csPhasePlane,
                              0.5*((double)draPhasePlane->allocation.width-d), 0);
     cairo_paint(cr);
     if((current_plus_dotsize>0.0)||(current_minus_dotsize>0.0))
       {phaseplane_draw_dots(cr, (double)widget->allocation.width,
                             (double)widget->allocation.height,
                             current_plus_dotsize, current_minus_dotsize);
       }
    }

  /*finish*/
  cairo_destroy (cr);
  
  return FALSE;
}

/*on clicked butRefresh*/
G_MODULE_EXPORT void cbRefresh (GtkButton *button, gpointer user_data)
{double desired_xmin, desired_xmax, desired_ymin, desired_ymax;
 double new_xmin, new_xmax, new_ymin, new_ymax;
 int desired_grid, new_grid;
 
 desired_xmin=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbXmin));
 desired_xmax=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbXmax));
 desired_ymin=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbYmin));
 desired_ymax=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbYmax));
 phaseplane_set_window( desired_xmin, desired_xmax, desired_ymin, desired_ymax );
 phaseplane_get_window( &new_xmin, &new_xmax, &new_ymin, &new_ymax );
 if(new_xmin!=desired_xmin)
   {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbXmin),(gdouble)new_xmin);}
 if(new_xmax!=desired_xmax)
   {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbXmax),(gdouble)new_xmax);}
 if(new_ymin!=desired_ymin)
   {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbYmin),(gdouble)new_ymin);}
 if(new_ymax!=desired_ymax)
   {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbYmax),(gdouble)new_ymax);}
 desired_grid=(int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbGridsize));
 phaseplane_set_dirfield_grid(desired_grid);
 new_grid=phaseplane_get_dirfield_grid();
 if(new_grid!=desired_grid)
   {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbGridsize),(gdouble)new_grid);}
 if(crPhasePlane)
   {double d=MINIMUM((double)draPhasePlane->allocation.width,(double)draPhasePlane->allocation.height);
    cairo_rectangle(crPhasePlane, 0.0, 0.0, d, d);
    cairo_clip(crPhasePlane);
    phaseplane_set_trajectory_color(1.0, 1.0, 1.0);
    phaseplane_set_draw_color_traj(1);
    phaseplane_set_background_color(0.0, 0.0, 0.0);
    phaseplane_set_axis_color(0.8, 0.8, 0.8);
    phaseplane_set_dirfield_color(0.0, 1.0, 1.0); 
    phaseplane_draw(crPhasePlane, d, d);
   }
 gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
}

/*on changed entYdot*/
G_MODULE_EXPORT void cbNewYdot (GtkCellEditable *celleditable, gpointer user_data)
{
 gtk_widget_set_sensitive(butSetEquation,TRUE);
}

/*on changed entXdot*/
G_MODULE_EXPORT void cbNewXdot (GtkCellEditable *celleditable, gpointer user_data)
{
 gtk_widget_set_sensitive(butSetEquation,TRUE);
}

/*on clicked butSetEquation*/
G_MODULE_EXPORT void cbSetEquation (GtkButton *button, gpointer user_data)
{int i;
 gchar *str, res[4096];

 /*set new equations with checking*/
 str=gtk_editable_get_chars( GTK_EDITABLE(entXdot), 0, -1);
 i=ode_set_equation(0, str);
 if(i!=0)
   {g_sprintf(messageStr,"Equation for x' is badly formatted.\nError realized at %dth character", (i<0)?(-i):i);
    error_message(messageStr);
    if(i==-1){ode_set_equation(0, "0.0" );}
   }
 g_free(str);
 ode_get_equation(0, res);
 gtk_entry_set_text( GTK_ENTRY(entXdot), res);
 
 str=gtk_editable_get_chars( GTK_EDITABLE(entYdot), 0, -1);
 i=ode_set_equation(1, str);
 if(i!=0)
   {g_sprintf(messageStr,"Equation for y' is badly formatted.\nError realized at %dth character", (i<0)?(-i):i);
    error_message(messageStr);
    if(i==-1){ode_set_equation(1, "0.0" );}
   }
 g_free(str);
 ode_get_equation(1, res);
 gtk_entry_set_text( GTK_ENTRY(entYdot), res);
 gtk_widget_set_sensitive(butSetEquation,FALSE);
 if(gtk_combo_box_get_active(GTK_COMBO_BOX(cmbTrajectory))!=-1)
   {gtk_widget_set_sensitive(butCalculate,TRUE);
    gtk_widget_set_sensitive(butCalculateAll,TRUE);
   }
}

/*on changed cmbParameter*/
G_MODULE_EXPORT void cbChangeParam (GtkComboBox *combobox, gpointer user_data)
{gint i;

 i=gtk_combo_box_get_active(GTK_COMBO_BOX(cmbParameter));
 if(i==-1)
   {/*editing*/
    editParam=TRUE;
   }
 else
   {/*new item choosed*/
    iParam=i;
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbParam), ode_get_parameter_value(i));
   }
}

/*on activate cmbentParameter*/
G_MODULE_EXPORT void cbModifyParam(GtkEntry *widget, gpointer user_data)
{const gchar *newtext;

 if(editParam)
   {/*editing of parameter name finished*/
    newtext=gtk_entry_get_text(GTK_ENTRY(cmbentParameter));
    if( ode_set_parameter_name(iParam, newtext) )
      {/*error to set new name*/
       gtk_combo_box_set_active( GTK_COMBO_BOX(cmbParameter), iParam );
      }
    else
      {/*set new name in combobox as well*/
       gtk_combo_box_remove_text(GTK_COMBO_BOX(cmbParameter), iParam);
       gtk_combo_box_insert_text(GTK_COMBO_BOX(cmbParameter), iParam, newtext);
       editParam=FALSE;
       gtk_widget_set_sensitive(butSetEquation,TRUE);
       gtk_combo_box_set_active( GTK_COMBO_BOX(cmbParameter), iParam );
      }
   }
}

/*on focus-out-event cmbentParameter*/
G_MODULE_EXPORT gboolean cbDefocusParam(GtkWidget *widget, GdkEventFocus *event, gpointer user_data)
{
 if(editParam)
   {/*editing cancelled*/
    gtk_combo_box_set_active( GTK_COMBO_BOX(cmbParameter), iParam );
    editParam=FALSE;
   }
 return FALSE;
}

/*on value-changed sbParam*/
G_MODULE_EXPORT void cbSetParam (GtkSpinButton *celleditable, gpointer user_data)
{double desired_value;
 int parami;
 
 parami=(int)gtk_combo_box_get_active(GTK_COMBO_BOX(cmbParameter));
 desired_value=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbParam));
 if( desired_value != ode_get_parameter_value(parami) )
   {ode_set_parameter_value( parami, desired_value );
    if(gtk_combo_box_get_active(GTK_COMBO_BOX(cmbTrajectory))!=-1)
      {gtk_widget_set_sensitive(butCalculate,TRUE);
       gtk_widget_set_sensitive(butCalculateAll,TRUE);
      }
   }
}

/*on clicked butAddTraj*/
G_MODULE_EXPORT void cbAddTraj (GtkButton *button, gpointer user_data)
{gchar str[32];
 
 g_sprintf(str, "traj%d", cumntraj);
 if( trajectory_add(str, cumntraj) )
   {error_message("Error: unable to add new trajectory.");
   }
 else
   {cumntraj++;
    gtk_combo_box_append_text( GTK_COMBO_BOX(cmbTrajectory), str );
    trajectory_set_initcond( str, 0, 1.0 );
    trajectory_set_initcond( str, 1, 1.0 );
    trajectory_set_tmin( str, -25.0 );
    trajectory_set_tmax( str, 25.0 );
    trajectory_set_dt( str, 0.01 );
    gtk_combo_box_set_active(GTK_COMBO_BOX(cmbTrajectory),ntraj);
    if(ntraj==0)
      {gtk_widget_set_sensitive(sbX0,TRUE);
       gtk_widget_set_sensitive(sbY0,TRUE);
       gtk_widget_set_sensitive(sbTmin,TRUE);
       gtk_widget_set_sensitive(sbTmax,TRUE);
       gtk_widget_set_sensitive(sbDt,TRUE);
      }
    gtk_widget_set_sensitive(butCalculate,TRUE);
    gtk_widget_set_sensitive(butCalculateAll,TRUE);
    ntraj++;
   }
}

/*on clicked butRemoveTraj*/
G_MODULE_EXPORT void cbRemoveTraj (GtkButton *button, gpointer user_data)
{gchar *name;
 gint iTraj;

 name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 iTraj=gtk_combo_box_get_active(GTK_COMBO_BOX(cmbTrajectory));
 if(iTraj>=0)
   {if( trajectory_delete(name) )
      {error_message("Error: unable to remove new trajectory.");
      }
    else
      {gtk_combo_box_remove_text(GTK_COMBO_BOX(cmbTrajectory), iTraj);
       ntraj--;
       if(ntraj){gtk_combo_box_set_active(GTK_COMBO_BOX(cmbTrajectory), MINIMUM(iTraj,ntraj-1));}
       else
         {gtk_widget_set_sensitive(sbX0,FALSE);
          gtk_widget_set_sensitive(sbY0,FALSE);
          gtk_widget_set_sensitive(sbTmin,FALSE);
          gtk_widget_set_sensitive(sbTmax,FALSE);
          gtk_widget_set_sensitive(sbDt,FALSE);
          gtk_widget_set_sensitive(butCalculate,FALSE);
          gtk_widget_set_sensitive(butCalculateAll,FALSE);
         }
       gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
      }
   }
 g_free(name);
}

/*on changed cmbTrajectory*/
G_MODULE_EXPORT void cbChangeTraj (GtkComboBox *combobox, gpointer user_data)
{gchar *name;
 GdkColor colTrajectory;
    
 name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 if(name==NULL){return;}
 if(name[0])
   {gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbX0), trajectory_get_initcond(name, 0) );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbY0), trajectory_get_initcond(name, 1) );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmin), trajectory_get_tmin(name) );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmax), trajectory_get_tmax(name) );
    if(!SameDt){gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbDt), trajectory_get_dt(name) );}
    gtk_widget_set_sensitive(butCalculate, (gboolean)trajectory_get_changed(name) );
    gtk_widget_set_sensitive(butCalculateAll, (gboolean)phaseplane_trajectory_changed() );
    trajectory_get_color(name, &colTrajectory);
    gtk_widget_modify_fg( GTK_WIDGET(draTrajectory), GTK_STATE_ACTIVE,  &colTrajectory);
    gtk_widget_queue_draw(GTK_WIDGET(draTrajectory));
   }
 g_free(name);
}

/*on expose-event draTrajectory*/
G_MODULE_EXPORT gboolean cbDrawTrajectoryColor (GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{
 gdk_draw_rectangle(widget->window,
                    widget->style->fg_gc[GTK_STATE_ACTIVE],
                    TRUE,
                    0, widget->allocation.height/4, 
                    widget->allocation.width, widget->allocation.height/2);
 return TRUE;
}

/*on value-changed sbX0*/
G_MODULE_EXPORT void cbSetX0(GtkSpinButton *celleditable, gpointer user_data)
{gchar *name;
 double desired_value;

 name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 desired_value=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbX0));
 if(desired_value!=trajectory_get_initcond(name, 0))
   {trajectory_set_initcond(name, 0, desired_value );
    gtk_widget_set_sensitive(butCalculate, TRUE);
    gtk_widget_set_sensitive(butCalculateAll, TRUE);
   }
 g_free(name);
}

/*on value-changed sbY0*/
G_MODULE_EXPORT void cbSetY0(GtkSpinButton *celleditable, gpointer user_data)
{gchar *name;
 double desired_value;

 name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 desired_value=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbY0));
 if(desired_value!=trajectory_get_initcond(name, 1))
   {trajectory_set_initcond(name, 1, desired_value );
    gtk_widget_set_sensitive(butCalculate, TRUE);
    gtk_widget_set_sensitive(butCalculateAll, TRUE);
   }
 g_free(name);
}

/*on value-changed sbTmin*/
G_MODULE_EXPORT void cbSetTmin(GtkSpinButton *celleditable, gpointer user_data)
{gchar *name;
 double desired_value;
 double new_value;

 name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 desired_value=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbTmin));
 if(desired_value!=trajectory_get_tmin(name))
   {trajectory_set_tmin(name,desired_value);
    new_value=trajectory_get_tmin(name);
    if(new_value!=desired_value)
      {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbTmin),(gdouble)new_value);}
    gtk_widget_set_sensitive(butCalculate, TRUE);
    gtk_widget_set_sensitive(butCalculateAll, TRUE);
   }
 g_free(name);
}

/*on value-changed sbTmax*/
G_MODULE_EXPORT void cbSetTmax(GtkSpinButton *celleditable, gpointer user_data)
{gchar *name;
 double desired_value;
 double new_value;

 name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 desired_value=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbTmax));
 if(desired_value!=trajectory_get_tmax(name))
   {trajectory_set_tmax(name,desired_value);
    new_value=trajectory_get_tmax(name);
    if(new_value!=desired_value)
      {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbTmax),(gdouble)new_value);}
    gtk_widget_set_sensitive(butCalculate, TRUE);
    gtk_widget_set_sensitive(butCalculateAll, TRUE);
   }
 g_free(name);
}

/*on value-changed sbDt*/
G_MODULE_EXPORT void cbSetDt(GtkSpinButton *celleditable, gpointer user_data)
{gchar *name;
 double desired_value;
 double new_value;
 
 desired_value=(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbDt));
 if(SameDt)
   {if( desired_value != ode_get_global_dt() )
      {ode_set_global_dt( desired_value );
       new_value=ode_get_global_dt();
       if(new_value!=desired_value)
         {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbDt),(gdouble)new_value);}
       gtk_widget_set_sensitive(butCalculate, TRUE);
       gtk_widget_set_sensitive(butCalculateAll, TRUE);
      }
   }
 else
   {name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
    if( desired_value != trajectory_get_dt(name) )
      {trajectory_set_dt(name,desired_value);
       new_value=trajectory_get_dt(name);
       if(new_value!=desired_value)
         {gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbDt),(gdouble)new_value);}
       gtk_widget_set_sensitive(butCalculate, TRUE);
       gtk_widget_set_sensitive(butCalculateAll, TRUE);
      }
    g_free(name);
   }
}

/*on changed chbSameDt*/
G_MODULE_EXPORT void cbChangedSameDt (GtkToggleButton *button, gpointer user_data)
{
 SameDt=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chbSameDt));
 if(SameDt)
   {ode_set_global_dt( gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbDt)) );
    gtk_widget_set_sensitive(butCalculate, TRUE);
    gtk_widget_set_sensitive(butCalculateAll, TRUE);
   }
 else
   {ode_set_global_dt( 0.0 );}
}

/*on clicked butCalculate*/
G_MODULE_EXPORT void cbCalculateTraj (GtkButton *button, gpointer user_data)
{gchar *name;
 double d;
 gchar estr[128];
 
 stop=0;
 gtk_widget_set_sensitive(butStop,TRUE);
 gtk_widget_set_sensitive(entXdot,FALSE);
 gtk_widget_set_sensitive(entYdot,FALSE);
 gtk_widget_set_sensitive(sbParam,FALSE);
 gtk_widget_set_sensitive(sbX0,FALSE);
 gtk_widget_set_sensitive(sbY0,FALSE);
 gtk_widget_set_sensitive(sbTmin,FALSE);
 gtk_widget_set_sensitive(sbTmax,FALSE);
 gtk_widget_set_sensitive(sbDt,FALSE);
 gtk_widget_set_sensitive(chbSameDt,FALSE);
 name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 if(name==NULL){return;}
 if(name[0])
   {/*calculating trajectory*/
    g_sprintf(stbStr, "Calculating %s.", name);
    gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, stbStr);
    if(!stop)
      {d=trajectory_calculate_plus(name);
       if(d>=0.0){gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmax), d);}
       else
         {g_sprintf(estr,"Unable to calculate the trajectory '%s' onwards. Try to decrease tmax or increase dt.",name);
          error_message(estr);
         }
      }
    if(!stop)
      {d=trajectory_calculate_minus(name);
       if(d<=0.0){gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmin), d);}
       else
         {g_sprintf(estr,"Unable to calculate the trajectory '%s' backwards. Try to increase tmin or increase dt.",name);
          error_message(estr);
         }
      }
    gtk_statusbar_pop(GTK_STATUSBAR(stbMain), stb_context);
    phaseplane_set_dirfield_grid(phaseplane_get_dirfield_grid());
    gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
   }
 gtk_widget_set_sensitive(butStop, FALSE);
 gtk_widget_set_sensitive(entXdot,TRUE);
 gtk_widget_set_sensitive(entYdot,TRUE);
 gtk_widget_set_sensitive(sbParam,TRUE);
 gtk_widget_set_sensitive(sbX0,TRUE);
 gtk_widget_set_sensitive(sbY0,TRUE);
 gtk_widget_set_sensitive(sbTmin,TRUE);
 gtk_widget_set_sensitive(sbTmax,TRUE);
 gtk_widget_set_sensitive(sbDt,TRUE);
 gtk_widget_set_sensitive(chbSameDt,TRUE);
 gtk_widget_set_sensitive(butCalculate, (gboolean)trajectory_get_changed(name) );
 gtk_widget_set_sensitive(butCalculateAll, (gboolean)phaseplane_trajectory_changed() );
 g_free(name);
}

/*on clicked butCalculateAll*/
G_MODULE_EXPORT void cbCalculateAll (GtkButton *button, gpointer user_data)
{gchar name[32], *name2, estr[128];
 int i;
 double d;
 
 stop=0;
 gtk_widget_set_sensitive(butStop,TRUE);
 gtk_widget_set_sensitive(entXdot,FALSE);
 gtk_widget_set_sensitive(entYdot,FALSE);
 gtk_widget_set_sensitive(sbParam,FALSE);
 gtk_widget_set_sensitive(sbX0,FALSE);
 gtk_widget_set_sensitive(sbY0,FALSE);
 gtk_widget_set_sensitive(sbTmin,FALSE);
 gtk_widget_set_sensitive(sbTmax,FALSE);
 gtk_widget_set_sensitive(sbDt,FALSE);
 gtk_widget_set_sensitive(chbSameDt,FALSE);
 for(i=0;i<cumntraj;i++)
    {g_sprintf(name, "traj%d", i);
     if(trajectory_get_changed(name)==0){continue;}
     /*calculating trajectory*/
     g_sprintf(stbStr, "Calculating %s.", name);
     gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, stbStr);
     if(!stop)
       {d=trajectory_calculate_plus(name);
        if(d<-1.0)
          {g_sprintf(estr,"Unable to calculate the trajectory '%s' onwards. Try to decrease tmax or increase dt.",name);
           error_message(estr);
          }
       }
     if(!stop)
       {d=trajectory_calculate_minus(name);
        if(d>1.0)
          {g_sprintf(estr,"Unable to calculate the trajectory '%s' backwards. Try to increase tmin or increase dt.",name);
           error_message(estr);
          }
       }
     gtk_statusbar_pop(GTK_STATUSBAR(stbMain), stb_context);
    }
 /*re-set tmin, tmax and butCalculate as they could change in the process of calculating*/
 name2=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 if(name2!=NULL)
   {if(name2[0])
      {gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmin), trajectory_get_tmin(name2) );
       gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmax), trajectory_get_tmax(name2) );
       gtk_widget_set_sensitive(butCalculate, (gboolean)trajectory_get_changed(name2) );
       gtk_widget_set_sensitive(butCalculateAll, (gboolean)phaseplane_trajectory_changed() );
      }
   }
 g_free(name2);
 /*to invoke drawing*/
 gtk_widget_set_sensitive(butStop,FALSE);
 gtk_widget_set_sensitive(entXdot,TRUE);
 gtk_widget_set_sensitive(entYdot,TRUE);
 gtk_widget_set_sensitive(sbParam,TRUE);
 gtk_widget_set_sensitive(sbX0,TRUE);
 gtk_widget_set_sensitive(sbY0,TRUE);
 gtk_widget_set_sensitive(sbTmin,TRUE);
 gtk_widget_set_sensitive(sbTmax,TRUE);
 gtk_widget_set_sensitive(sbDt,TRUE);
 gtk_widget_set_sensitive(chbSameDt,TRUE);
 phaseplane_set_dirfield_grid(phaseplane_get_dirfield_grid());
 gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
}

/*on value-changed sbTunit*/
G_MODULE_EXPORT void cbSetTunit(GtkSpinButton *celleditable, gpointer user_data)
{
 animDt=1.0/(double)gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbTunit));
 if(animDt<ANIM_MIN_TUNIT)
   {animDt=ANIM_MIN_TUNIT;
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbTunit),(gdouble)animDt);
   }
 if(animDt>ANIM_MAX_TUNIT)
   {animDt=ANIM_MAX_TUNIT;
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbTunit),(gdouble)animDt);
   }
 /*using fps, set it to Dt for 1 frame*/
 animDt/=ANIM_FPS;
 return;
}

/*******************************/
/*function to call periodically*/
/*******************************/
gboolean funAnimate(gpointer data)
{double startplus=0.0, startminus=0.0;
 
 gtk_statusbar_pop(GTK_STATUSBAR(stbMain), stb_context);
 switch(Anim_Type)
   {case 0:
       {if(animplusi<=animplus)
          {phaseplane_set_time_frame(0.0, animDt*animplusi);
           startplus=animDt*(animplusi-1);
           startminus=0.0;
           g_sprintf(stbStr, "Time: %.6f/%g", animDt*animplusi, maxtmax);
           gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, stbStr);
           current_minus_dotsize=0.0;
           if(Flash_Freq==0)
             {current_plus_dotsize=ANIM_DOTSIZE;}
           else
             {if( (animplusi/Flash_Freq)%2 )
                {current_plus_dotsize=0.0;}
              else
                {current_plus_dotsize=ANIM_DOTSIZE;}
             }
           animplusi++;
          }
        else if(animminusi<=animminus)
               {phaseplane_set_time_frame(-animDt*animminusi, maxtmax);
                startplus=maxtmax;
                startminus=-animDt*(animminusi-1);
                g_sprintf(stbStr, "Time: %.6f/%g", -animDt*animminusi, mintmin);
                gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, stbStr);
                current_plus_dotsize=0.0;
                if(Flash_Freq==0)
                  {current_minus_dotsize=ANIM_DOTSIZE;}
                else
                  {if( (animminusi/Flash_Freq)%2 )
                     {current_minus_dotsize=0.0;}
                   else
                     {current_minus_dotsize=ANIM_DOTSIZE;}
                  }
                animminusi++;
               }
        break;
       }/*end of case 0:*/
    case 1:
       {phaseplane_set_time_frame(-animDt*animplusi, animDt*animplusi);
        startplus=animDt*(animplusi-1);
        startminus=-animDt*(animminusi-1);
        if(Flash_Freq==0)
          {current_plus_dotsize=ANIM_DOTSIZE;
           current_minus_dotsize=ANIM_DOTSIZE;
          }
        else
          {if( (animplusi/Flash_Freq)%2 )
             {current_plus_dotsize=0.0;
              current_minus_dotsize=0.0;
             }
           else
             {current_plus_dotsize=ANIM_DOTSIZE;
              current_minus_dotsize=ANIM_DOTSIZE;
             }
          }
        g_sprintf(stbStr, "Time: %.6f/%g", animDt*animplusi, MAXIMUM(maxtmax,-mintmin) );
        gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, stbStr);
        animplusi++;
        break;
       }
   }/*end of switch(Anim_Type)*/
 if(crPhasePlane)
   {double d=MINIMUM((double)draPhasePlane->allocation.width,(double)draPhasePlane->allocation.height);
    phaseplane_draw_delta(crPhasePlane, d, d, startplus, startminus);
   }
 gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
 /*finishing animation*/
 if( ( (animplusi>animplus)&&(animminusi>animminus) ) || (stop!=0) )
   {if(crPhasePlane)
      {cairo_destroy(crPhasePlane);
       crPhasePlane=NULL;
       cairo_surface_destroy(csPhasePlane);
       csPhasePlane=NULL;
      }
    phaseplane_set_time_frame(-MIN_DT*INT_MAX, MIN_DT*INT_MAX);
    gtk_statusbar_pop(GTK_STATUSBAR(stbMain), stb_context);
    gtk_widget_queue_draw(GTK_WIDGET(draPhasePlane));
    gtk_widget_set_sensitive(butStop,FALSE);
    current_plus_dotsize=0.0;
    current_minus_dotsize=0.0;
    gtk_widget_set_sensitive(entXdot,TRUE);
    gtk_widget_set_sensitive(entYdot,TRUE);
    gtk_widget_set_sensitive(sbParam,TRUE);
    gtk_widget_set_sensitive(sbTunit,TRUE);
    gtk_widget_set_sensitive(sbX0,TRUE);
    gtk_widget_set_sensitive(sbY0,TRUE);
    gtk_widget_set_sensitive(sbTmin,TRUE);
    gtk_widget_set_sensitive(sbTmax,TRUE);
    gtk_widget_set_sensitive(sbDt,TRUE);
    gtk_widget_set_sensitive(chbSameDt,TRUE);
    return FALSE;
   }
 else {return TRUE;}
}

/*on clicked butAnimate*/
G_MODULE_EXPORT void cbAnimate (GtkButton *button, gpointer user_data)
{gchar name[32], *name2, estr[128];
 int i;
 double d;
 
 stop=0;
 gtk_widget_set_sensitive(butStop,TRUE);
 gtk_widget_set_sensitive(entXdot,FALSE);
 gtk_widget_set_sensitive(entYdot,FALSE);
 gtk_widget_set_sensitive(sbParam,FALSE);
 gtk_widget_set_sensitive(sbTunit,FALSE);
 gtk_widget_set_sensitive(sbX0,FALSE);
 gtk_widget_set_sensitive(sbY0,FALSE);
 gtk_widget_set_sensitive(sbTmin,FALSE);
 gtk_widget_set_sensitive(sbTmax,FALSE);
 gtk_widget_set_sensitive(sbDt,FALSE);
 gtk_widget_set_sensitive(chbSameDt,FALSE);
 /*first calculate trajectories, if necessary*/
 mintmin=0.0;
 maxtmax=0.0;
 for(i=0;i<cumntraj;i++)
    {g_sprintf(name, "traj%d", i);
     d=trajectory_get_tmax(name);
     if(d>maxtmax){maxtmax=d;}
     d=trajectory_get_tmin(name);
     if(d<mintmin){mintmin=d;}
     if(trajectory_get_changed(name)==0){continue;}
     /*calculating trajectory*/
     g_sprintf(stbStr, "Calculating %s.", name);
     gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, stbStr);
     if(!stop)
       {d=trajectory_calculate_plus(name);
        if(d<-1.0)
          {g_sprintf(estr,"Unable to calculate the trajectory '%s' onwards. Try to decrease tmax or increase dt.",name);
           error_message(estr);
          }
        else if(d>maxtmax){maxtmax=d;}
       }
     if(!stop)
       {d=trajectory_calculate_minus(name);
        if(d>1.0)
          {g_sprintf(estr,"Unable to calculate the trajectory '%s' backwards. Try to increase tmin or increase dt.",name);
           error_message(estr);
          }
        else if(d<mintmin){mintmin=d;}
       }
     gtk_statusbar_pop(GTK_STATUSBAR(stbMain), stb_context);
    }
 /*re-set tmin, tmax and butCalculate as they could change in the process of calculating*/
 name2=gtk_combo_box_get_active_text(GTK_COMBO_BOX(cmbTrajectory));
 if(name2!=NULL)
   {if(name2[0])
      {gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmin), trajectory_get_tmin(name2) );
       gtk_spin_button_set_value( GTK_SPIN_BUTTON(sbTmax), trajectory_get_tmax(name2) );
       gtk_widget_set_sensitive(butCalculate, (gboolean)trajectory_get_changed(name2) );
       gtk_widget_set_sensitive(butCalculateAll, (gboolean)phaseplane_trajectory_changed() );
      }
   }
 g_free(name2);
 phaseplane_set_dirfield_grid(phaseplane_get_dirfield_grid());
 /*animate*/
 animplus=(int)(maxtmax/animDt);
 animminus=-(int)(mintmin/animDt);
 animplusi=0;
 phaseplane_set_time_frame(0.0, 0.0);
 
 /*create a buffer cairo*/
 d=MINIMUM((double)draPhasePlane->allocation.width,(double)draPhasePlane->allocation.height);
 csPhasePlane=cairo_image_surface_create(CAIRO_FORMAT_RGB24, d, d);
 if(cairo_surface_status(csPhasePlane)!=CAIRO_STATUS_SUCCESS)
   {cairo_surface_destroy(csPhasePlane);
    csPhasePlane=NULL;
    crPhasePlane=NULL;
	g_warning("Unable to create cairo_image_surface, fall back to direct drawing.");
   }
 else
   {crPhasePlane=cairo_create(csPhasePlane);
    if(cairo_status(crPhasePlane)!=CAIRO_STATUS_SUCCESS)
      {cairo_destroy(crPhasePlane);
       crPhasePlane=NULL;
       cairo_surface_destroy(csPhasePlane);
       csPhasePlane=NULL;
	   g_warning("Unable to create cairo_image_surface, fall back to direct drawing.");
      }
   }
 
 switch(Anim_Type)
   {case 0:
       {g_sprintf(stbStr, "Time: %.6f/%g", 0.0, maxtmax);
        animminusi=0;
        break;
       }
    case 1:
       {g_sprintf(stbStr, "Time: %.6f/%g", 0.0, MAXIMUM(maxtmax,-mintmin) );
        animminusi=animminus+1;
        animplus=MAXIMUM(animplus, animminus);
        break;
       }
   }
 /*initial drawing*/
 if(crPhasePlane)
   {cairo_rectangle(crPhasePlane, 0.0, 0.0, d, d);
    cairo_clip(crPhasePlane);
    phaseplane_set_trajectory_color(1.0, 1.0, 1.0);
    phaseplane_set_draw_color_traj(1);
    phaseplane_set_background_color(0.0, 0.0, 0.0);
    phaseplane_set_axis_color(0.8, 0.8, 0.8);
    phaseplane_set_dirfield_color(0.0, 1.0, 1.0); 
    phaseplane_draw(crPhasePlane, d, d);
   }
 gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, stbStr);
 g_timeout_add((guint)(1000.0/ANIM_FPS), funAnimate, NULL);
 return;
}

/*on clicked butStop*/
G_MODULE_EXPORT void cbStop (GtkButton *button, gpointer user_data)
{
 trajectory_calculation_stop();
 stop=1;
}


/***********/
/* M A I N */
/***********/

int main(int argc, char *argv[])
{GladeXML *xml;
 GtkWidget *mainWindow;
 int i;

 /*initialisation*/
 #if GTK_MINOR_VERSION>=10
 PrintSettings=NULL;
 #endif
 editParam=FALSE;
 editTraj=FALSE;
 editXdot=FALSE; editYdot=FALSE;
 gtk_init (&argc, &argv);
 glade_init();

 /* load the interface */
 xml = glade_xml_new("phasepictor_main.glade", NULL, NULL);
 /* connect the signals in the interface */
 glade_xml_signal_autoconnect(xml);
 
 /*get important widgets*/
 mainWindow=glade_xml_get_widget(xml,"mainWindow");
 stbMain=glade_xml_get_widget(xml,"stbMain");
 stb_context=gtk_statusbar_get_context_id(GTK_STATUSBAR(stbMain), "General context.");
 gtk_statusbar_push(GTK_STATUSBAR(stbMain), stb_context, "Done.");
 menuBar=glade_xml_get_widget(xml,"menuBar");
 draPhasePlane=glade_xml_get_widget(xml,"draPhasePlane");
 expPhasePlane=glade_xml_get_widget(xml,"expPhasePlane");
 expEquation=glade_xml_get_widget(xml,"expEquation");
 expParameter=glade_xml_get_widget(xml,"expParameter");
 expTrajectory=glade_xml_get_widget(xml,"expTrajectory");
 tbFullscreen=glade_xml_get_widget(xml,"tbFullscreen");
 sbXmin=glade_xml_get_widget(xml,"sbXmin");
 sbXmax=glade_xml_get_widget(xml,"sbXmax");
 sbYmin=glade_xml_get_widget(xml,"sbYmin");
 sbYmax=glade_xml_get_widget(xml,"sbYmax");
 sbGridsize=glade_xml_get_widget(xml,"sbGridsize");
 chbPrintColors=glade_xml_get_widget(xml,"chbPrintColors");
 entXdot=glade_xml_get_widget(xml,"entXdot");
 entYdot=glade_xml_get_widget(xml,"entYdot");
 butSetEquation=glade_xml_get_widget(xml,"butSetEquation");
 cmbParameter=glade_xml_get_widget(xml,"cmbParameter");
 cmbentParameter=glade_xml_get_widget(xml,"cmbentParameter");
 iParam=0;
 gtk_combo_box_set_active(GTK_COMBO_BOX(cmbParameter),iParam);
 sbParam=glade_xml_get_widget(xml,"sbParam");
 cmbTrajectory=glade_xml_get_widget(xml,"cmbTrajectory");
 draTrajectory=glade_xml_get_widget(xml,"draTrajectory");
 sbX0=glade_xml_get_widget(xml,"sbX0");
 gtk_widget_set_sensitive(sbX0,FALSE);
 sbY0=glade_xml_get_widget(xml,"sbY0");
 gtk_widget_set_sensitive(sbY0,FALSE);
 sbTmin=glade_xml_get_widget(xml,"sbTmin");
 gtk_widget_set_sensitive(sbTmin,FALSE);
 sbTmax=glade_xml_get_widget(xml,"sbTmax");
 gtk_widget_set_sensitive(sbTmax,FALSE);
 sbDt=glade_xml_get_widget(xml,"sbDt");
 gtk_widget_set_sensitive(sbDt,FALSE);
 chbSameDt=glade_xml_get_widget(xml,"chbSameDt");
 SameDt=TRUE;
 gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chbSameDt), TRUE);
 butCalculate=glade_xml_get_widget(xml,"butCalculate");
 gtk_widget_set_sensitive(butCalculate,FALSE);
 butCalculateAll=glade_xml_get_widget(xml,"butCalculateAll");
 gtk_widget_set_sensitive(butCalculateAll,FALSE);
 expAnimation=glade_xml_get_widget(xml,"expAnimation");
 sbTunit=glade_xml_get_widget(xml,"sbTunit");
 butAnimate=glade_xml_get_widget(xml,"butAnimate");
 butStop=glade_xml_get_widget(xml,"butStop");
 gtk_widget_set_sensitive(butStop,FALSE);
 tbPrint=glade_xml_get_widget(xml,"tbPrint");
 
 #if GTK_MINOR_VERSION<10
 gtk_widget_hide(tbPrint);
 #endif
 
 /*important widgets of the New model dialog*/
 dlgNewModel=glade_xml_get_widget(xml,"dlgNewModel");
 gtk_window_set_title(GTK_WINDOW(dlgNewModel), "New model");
 sbNumVariables=glade_xml_get_widget(xml,"sbNumVariables");
 gtk_spin_button_set_range(GTK_SPIN_BUTTON(sbNumVariables), 2.0, MAX_VALT-2.0);
 gtk_widget_set_sensitive(GTK_WIDGET(sbNumVariables), FALSE);
 sbNumParameters=glade_xml_get_widget(xml,"sbNumParameters");
 gtk_spin_button_set_range(GTK_SPIN_BUTTON(sbNumParameters), 1.0, MAX_VALT-2.0);
 
 /*create file filters*/
 allFilter=gtk_file_filter_new();
 gtk_file_filter_set_name(allFilter,(gchar *)"All files (*.*)");
 gtk_file_filter_add_pattern(allFilter,(gchar *)"*");
 pplFilter=gtk_file_filter_new();
 gtk_file_filter_set_name(pplFilter,(gchar *)"Saved phaseplane (*.ppl)");
 gtk_file_filter_add_pattern(pplFilter,(gchar *)"*.ppl");
  
 /*create a file chooser dialog for Open*/
 dialogOpen = gtk_file_chooser_dialog_new ("Open phaseplane",
                                       GTK_WINDOW(mainWindow),
                                       GTK_FILE_CHOOSER_ACTION_OPEN,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                       GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                       (const gchar *)NULL);
 gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER(dialogOpen), TRUE);
 gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialogOpen), pplFilter);
 gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialogOpen), allFilter);
 gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialogOpen), pplFilter);
 gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialogOpen), FALSE);
 filenameSave=0;
 folderExport=0;
 
 /*initialize the numeric part*/ 
 if( phaseplane_init(ode_dimension,ode_nparam) )
   {g_error("Unable to initialize phaseplane.\n");}
 for(i=0;i<MINIMUM(ode_nparam,4);i++)
    {g_sprintf(par_name,"%c",i+97);
     ode_set_parameter_name(i,par_name);
     ode_set_parameter_value(i,1.0);
     gtk_combo_box_append_text( GTK_COMBO_BOX(cmbParameter), par_name );
    }
 for(;i<ode_nparam;i++)
    {g_sprintf(par_name,"p%d",i);
     ode_set_parameter_name(i,par_name);
     ode_set_parameter_value(i,1.0);
     gtk_combo_box_append_text( GTK_COMBO_BOX(cmbParameter), par_name );
    }
 gtk_combo_box_set_active( GTK_COMBO_BOX(cmbParameter), 0);
 ode_set_variable_name(0, "x"); ode_set_variable_name(1, "y");  
 ode_set_equation(0, gtk_editable_get_chars( GTK_EDITABLE(entXdot), 0, -1) );
 ode_set_equation(1, gtk_editable_get_chars( GTK_EDITABLE(entYdot), 0, -1) );
 ode_set_global_dt( gtk_spin_button_get_value(GTK_SPIN_BUTTON(sbDt)) );
 
 /*initialize kparser*/
 init_kparser();
 add_comment_character('#');
 add_whitespace_character(' ');
 add_whitespace_character('\n');
 add_whitespace_character('\r');
 add_whitespace_character('\t');
 
 /*@later: process parameterfile, initialize global variables*/
 animDt=1/ANIM_FPS;
 gtk_spin_button_set_value(GTK_SPIN_BUTTON(sbTunit),(gdouble)animDt*ANIM_FPS);
 
 /* start the event loop */
 gtk_main();
/* gtk_widget_show (mainWindow);*/
 
 phaseplane_destroy();
 return 0;
}
